/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Theme;
using Microsoft.Practices.Prism.Commands;
using PilotShare.Client.Properties;
using PilotShare.Client.ToolbarItems;
using PilotShare.Client.Utils;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace PilotShare.Client.View
{
    class LinksManagerViewModel : PropertyChangedBase, IDisposable
    {
        private readonly IDataObject _target;
        private readonly ISearchService _searchService;
        private readonly IObjectsRepository _repository;
        private readonly ILinkObjectModifier _modifier;
        private readonly ISettingsService _settingsService;
        private readonly IPilotShareDialogService _dialogService;

        private DelegateCommand _createLinkCommand;
        private DelegateCommand _editLinkCommand;
        private DelegateCommand _removeLinkCommand;
        private DelegateCommand _copyToClipboardCommand;
        private DelegateCommand _openInBrowserCommand;

        private ObservableCollection<object> _toolbarCommands;
        private LinkItem _selectedLink;
        
        public LinksManagerViewModel(IDataObject target, ISearchService searchService, 
            IObjectsRepository repository, ILinkObjectModifier modifier, 
            ISettingsService settingsService, IPilotShareDialogService dialogService)
        {
            _target = target;
            _searchService = searchService;
            _repository = repository;
            _modifier = modifier;
            _settingsService = settingsService;
            _dialogService = dialogService;

            LinkCollection = new ObservableCollection<LinkItem>();
            _ = InitLinks(repository, searchService);
            InitToolbarCommands();
        }

        public ObservableCollection<object> ToolbarCommands => _toolbarCommands;

        public ObservableCollection<LinkItem> LinkCollection { get; }

        public ICommand EditLinkCommand => _editLinkCommand;
        public ICommand CopyCommand => _copyToClipboardCommand;
        public ICommand CreateLinkCommand => _createLinkCommand;
        public ICommand RemoveLinkCommand => _removeLinkCommand;
        public ICommand OpenInBrowserCommand => _openInBrowserCommand;


        public LinkItem SelectedLink
        {
            get => _selectedLink;
            set
            {
                _selectedLink = value;
                NotifyOfPropertyChange(nameof(SelectedLink));

                _editLinkCommand.RaiseCanExecuteChanged();
                _removeLinkCommand.RaiseCanExecuteChanged();
            }
        }


        public void Dispose()
        {
        }

        private void InitToolbarCommands()
        {
            _createLinkCommand = new DelegateCommand(CreateLink)
            {
                Caption = Resources.CreateLink,
                Hint = Resources.CreateLinkHint,
                Icon = Icons.Instance.CreateIcon
            };

            _editLinkCommand = new DelegateCommand(EditLink, CanEdit)
            {
                Caption = Resources.EditLink,
                Hint = Resources.EditLinkHint,
                Icon = Icons.Instance.EditIcon
            };

            _removeLinkCommand = new DelegateCommand(RemoveLink, CanEdit)
            {
                Caption = Resources.RemoveLink,
                Hint = Resources.RemoveLinkHint,
                Icon = Icons.Instance.DeleteIcon
            };

            _toolbarCommands = new ObservableCollection<object>
            {
                new ToolbarButtonItem("AUIDCreateLink").WithCommand(_createLinkCommand),
                new ToolbarButtonItem("AUIDEditLink").WithCommand(_editLinkCommand),
                new ToolbarButtonItem("AUIDRemoveLink").WithCommand(_removeLinkCommand)
            };

            _copyToClipboardCommand = new DelegateCommand(CopyToClipboard);
            _openInBrowserCommand = new DelegateCommand(OpenInBrowser);

            NotifyOfPropertyChange(nameof(ToolbarCommands));
        }

        private void OpenInBrowser()
        {
            if (_selectedLink == null)
                return;

            System.Diagnostics.Process.Start(_selectedLink.Url);
        }

        private void CopyToClipboard()
        {
            if (_selectedLink == null)
                return;

            Clipboard.SetText(_selectedLink.Url);
        }

        private bool CanEdit()
        {
            return _selectedLink != null;
        }

        private void RemoveLink()
        {
            _modifier.RemoveLink(_selectedLink.Id);

            LinkCollection.Remove(_selectedLink);
        }

        private void EditLink()
        {
            var itemToEdit = _selectedLink.Clone();
            if (_dialogService.ShowEditLinkDialog(itemToEdit) == true)
            {
                var indexOf = LinkCollection.IndexOf(SelectedLink);
                LinkCollection.Remove(SelectedLink);
                SelectedLink = itemToEdit;
                LinkCollection.Insert(indexOf, SelectedLink);
            }
        }

        private async void CreateLink()
        {
            var linkId = Guid.NewGuid();
            var url = await _settingsService.LoadLinkUrl(linkId);
            var link = new LinkItem(linkId, url);
            if (_dialogService.ShowCreateLinkDialog(_target, link) == true)
            {
                LinkCollection.Add(link);
            }
        }

        private async Task InitLinks(IObjectsRepository repository, ISearchService searchService)
        {
            LinkCollection.Clear();

            var externalLinkType = repository.GetType(Const.EXTERNAL_LINK_TYPE_NAME);
            if (externalLinkType == null)
                return;

            var linksLoaderService = new LinksLoaderService(searchService, externalLinkType);
            var ids = await linksLoaderService.LoadLinksAsync(_target);

            var loader = new ObjectLoader(repository);
            var links = await loader.LoadObjectsAsync(ids.ToArray());

            foreach (var link in links)
            {
                var linkUrl = await _settingsService.LoadLinkUrl(link.Id);
                var item = new LinkItem(link.Id, linkUrl);

                if (link.Attributes.TryGetValue(LinkAttributeNames.AUTHOR, out var authorAttr))
                {
                    if (int.TryParse(authorAttr.ToString(), out var authorId))
                    {
                        var person = repository.GetPerson(authorId);
                        item.SharedBy = person.DisplayName;
                    }
                }

                if (link.Attributes.TryGetValue(LinkAttributeNames.PASSWORD, out var passwordAttr))
                {
                    if (passwordAttr != null)
                    {
                        item.Password = passwordAttr.ToString();
                    }
                }

                if (link.Attributes.TryGetValue(LinkAttributeNames.DATE, out var dateAttr))
                {
                    if (DateTime.TryParse(dateAttr.ToString(), out var date))
                    {
                        item.Date = date;
                    }
                }

                if (link.Attributes.TryGetValue(LinkAttributeNames.EXPIRATION_DATE, out var expDateAttr))
                {
                    if (DateTime.TryParse(expDateAttr.ToString(), out var date))
                    {
                        item.ValidThrough = date;
                    }
                }

                LinkCollection.Add(item);
            }
        }
    }

    public class LinkItem
    {
        public LinkItem(Guid id, string url)
        {
            Id = id;
            Url = url;
        }

        public Guid Id { get; }
        public string Url { get; }
        public DateTime? ValidThrough { get; set; }
        public string SharedBy { get; set; }
        public DateTime Date { get; set; }
        public string Password { get; set; }

        public LinkItem Clone()
        {
            return new LinkItem(Id, Url)
            {
                Date = Date,
                Password = Password,
                SharedBy = SharedBy,
                ValidThrough = ValidThrough
            };
        }
    }
}
