#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <sys/socket.h>
#include <gio/gunixsocketaddress.h>
#include <libcaja-extension/caja-extension-types.h>
#include <libcaja-extension/caja-file-info.h>
#include <libcaja-extension/caja-menu-provider.h>
#include <libcaja-extension/caja-info-provider.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/un.h>
#include <pthread.h>
#include <locale.h>
#include "Commands/Proto/protobuf-c/protobuf-c.h"
#include "Commands/Proto/DataContract.pb-c.h"
#include "Commands/Utils.h"
#include <poll.h>

#define FILE_URI_PREFIX "file://"
#define ADDRESS_FOR_NOTIF_STATE "/tmp/CoreFxPipe_IconStateNotifierPipe"
#define ADDRESS_FOR_REQ_STATE "/tmp/CoreFxPipe_PilotInfoPipepfmfs.3D-Storage20170614"
#define ADDRESS_FOR_COMMAND "/tmp/CoreFxPipe_PilotCommandPipepfmfs.3D-Storage20170614"
#define RECV_MESSAGE_LENGTH 1000
#define VERSION "25.10.0.0"
#define UNIX_PATH_MAX 108
/*\
|*|
|*| BUILD SETTINGS
|*|
\*/

#ifdef ENABLE_NLS
#include <glib/gi18n-lib.h>
#include <sys/types.h>
#define I18N_INIT() \
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
#else
#define _(STRING) ((char * ) STRING)
#define I18N_INIT()
#endif


/*\
|*|
|*| GLOBAL TYPES AND VARIABLES
|*|
\*/

int currentEmblem = 0;
static char* emblems[] = {
		"", "emblem-loadedstore", "emblem-outdatedstore", "emblem-notsentstore", "emblem-editedstore",
		"emblem-abortedstore", "", "",
		"emblem-conflictstore"
};

static GHashTable *pending_updates = NULL;
static GType provider_types[1];
static GType caja_3dstorage_extension_type;
static GObjectClass* parent_class;

typedef struct FileStateInfo {
	char* uri;
	int icon_index;
} FileStateInfo;

/*\
|*|
|*| FUNCTIONS
|*|
\*/
char* ReadString(FILE* file);

char* PrependStringLength(const char* string);

char* GetFilePath(CajaFileInfo fileInfo);

FileStateInfo GetFileInfo(FILE* pipe_file);
FileStateInfo ParseFileInfo(const unsigned char* buffer, ssize_t total_len);

char* AddFilePrefix(const char* path_to_file);

DataContracts__CommandInvokeResult SerializeAndSend(DataContracts__CommandInvokeData instance);

static void CommandInvoke(CajaMenuItem* item);
static struct FilesPathToCommand GetFilePathsToCommand(GList* files);

int ChangeFileEmblem(CajaFileInfo* file, int icon_index);
static void FreeArray(char** array, int itemCount);

struct FilesPathToCommand
{
	int count;
	char ** Paths;
};

char* ReadString(FILE* file) {
	int lenght;
	const int char_count = 256;
	unsigned char byte[2];

	byte[0] = fgetc(file);
	byte[1] = fgetc(file);

	lenght = byte[0] * char_count + byte[1];
	unsigned char bytes[lenght];

	int i = 0;
	size_t bytes_read = fread(bytes, 1, lenght, file);
	if (bytes_read != lenght) {
		printf(">>> ReadString error \n");
	}

	char* chars = (char*)malloc(lenght + 1);
	memcpy(chars, bytes, lenght);
	chars[lenght] = '\0';
	return chars;
}

char* PrependStringLength(const char* string) {
	const int stringLength = strlen(string);
	const char stringLengthFirstByte = (stringLength / 255);
	const char stringLengthSecondByte = (stringLength % 255);

	char* resultString = malloc(2 + stringLength + 1);
	memcpy(resultString, &stringLengthFirstByte, sizeof(char));
	memcpy(resultString + sizeof(char), &stringLengthSecondByte, sizeof(char));
	memcpy(resultString + sizeof(char) + sizeof(char), string, stringLength);
	resultString[2 + stringLength] = '\0';
	return resultString;
}

FileStateInfo GetFileInfo(FILE* pipe_file) {
	FileStateInfo file_info = {NULL, 0};
	for (int b = 0; b < 2; b++) {
		char *chars = ReadString(pipe_file);

		if (strlen(chars) > 2) {
			char* raw_uri = AddFilePrefix(chars);
			GFile* file = g_file_new_for_uri(raw_uri);
			file_info.uri = g_file_get_uri(file);  // Получаем нормализованный URI
			g_object_unref(file);
			free(raw_uri);  // Освобождаем промежуточную строку
		} 		else
			file_info.icon_index = atoi(chars);
		free(chars);
	}
	printf(">>> result uri - %s \n", file_info.uri);
	printf(">>> result index - %i \n", file_info.icon_index);

	return file_info;
}

FileStateInfo ParseFileInfo(const unsigned char* buffer, ssize_t total_len) {
	FileStateInfo info = {NULL, 0};

	// Проверяем минимальную длину (2 байта длины + 1 байт иконки)
	if (total_len < 3) {
		fprintf(stderr, "Слишком короткое сообщение: %zd байт\n", total_len);
		return info;
	}

	uint16_t path_len = (buffer[0] << 8) | buffer[1] - 1;
	// Проверяем корректность длины
	if (path_len == 0 || path_len > (total_len - 2)) {
		fprintf(stderr, "Некорректная длина пути: %u (всего данных: %zd)\n",
			   path_len, total_len);
		return info;
	}

	char* path = malloc(path_len + 1);
	if (!path) {
		perror("malloc failed");
		return info;
	}
	memcpy(path, buffer + 2, path_len);
	path[path_len] = '\0';
	char* uri = g_filename_to_uri(path, NULL, NULL);
	free(path);

	if (!uri) {
		fprintf(stderr, "Ошибка преобразования пути в URI\n");
		return info;
	}

	info.uri = uri;
	fprintf(stderr, ">>>>>> Cообщение: %s \n", info.uri);
	char index_str[2] = {buffer[2 + path_len], '\0'};
	info.icon_index = atoi(index_str);
	fprintf(stderr, ">>>>> Icon: %i \n", info.icon_index);

	return info;
}

char* AddFilePrefix(const char* path_to_file) {
	char* uri = malloc(strlen(FILE_URI_PREFIX) + strlen(path_to_file) + 1);
	memcpy(uri, FILE_URI_PREFIX, strlen(FILE_URI_PREFIX));
	memcpy(uri + strlen(FILE_URI_PREFIX), path_to_file, strlen(path_to_file) + 1);
	return uri;
}

static int RequestState(const char* path) {
	GError *error = NULL;
	gsize bytes_read = 0;
	gchar recv_msg[RECV_MESSAGE_LENGTH] = {0};
	int result = 0;

	char* PathWithStringLength = PrependStringLength(path);
	if (!PathWithStringLength) {
		g_warning("Failed to prepare path string");
		return 0;
	}

	const int sizeOfTwoByteForStrLength = sizeof(char) * 2;
	const int sizeOfTerminalNull = 1;
	const int sendMessageLength = sizeOfTwoByteForStrLength + strlen(PathWithStringLength + sizeOfTwoByteForStrLength) + sizeOfTerminalNull;

	GSocket *socket_for_requests = g_socket_new(G_SOCKET_FAMILY_UNIX,
											  G_SOCKET_TYPE_STREAM,
											  G_SOCKET_PROTOCOL_DEFAULT,
											  &error);
	if (!socket_for_requests) {
		g_warning("Failed to create socket: %s", error->message);
		g_error_free(error);
		free(PathWithStringLength);
		return 0;
	}

	printf("Client: Trying to connect... \n");

	printf("Socket path: %s\n", ADDRESS_FOR_REQ_STATE);
	if (!G_IS_SOCKET(socket_for_requests)) {
		g_warning("Socket is invalid");
		return 0;
	}

	GSocketAddress *address = g_unix_socket_address_new(ADDRESS_FOR_REQ_STATE);
	if (!address) {
		g_warning("Failed to create socket address");
		g_object_unref(socket_for_requests);
		free(PathWithStringLength);
		return 0;
	}


	g_socket_set_timeout(socket_for_requests, 5);
	printf("adress correct t : ");
	if (!g_socket_connect(socket_for_requests, address, NULL, &error)) {
		printf("warn \n");
		g_warning("Connection failed: %s", error ? error->message : "Unknown error");
		if (error)
		{
		g_error_free(error);
		g_object_unref(address);
		g_object_unref(socket_for_requests);
		free(PathWithStringLength);
		return 0;
		}
	}
	g_object_unref(address);

	printf("Client: Connected \n");



	if (!g_socket_send(socket_for_requests, PathWithStringLength, sendMessageLength, NULL, &error)) {
		g_warning("Failed to send data: %s", error->message);
		g_error_free(error);
		g_object_unref(socket_for_requests);
		return 0;
	}

	free(PathWithStringLength);
	PathWithStringLength = NULL;


	GInputVector vector = { recv_msg, RECV_MESSAGE_LENGTH };
	gssize bytes_received = g_socket_receive_message(socket_for_requests, NULL, &vector, 1, NULL, NULL, NULL, NULL, &error);
	if (bytes_received > 0) {
		printf("Client: Data received (%ld bytes): %s\n", (long)bytes_received, recv_msg);
		result = atoi(recv_msg);
	} else {
		if (error) {
			g_warning("Client: Error on receive: %s", error->message);
			g_error_free(error);
		} else {
			printf("Client: Server socket closed \n");
		}
	}

	g_object_unref(socket_for_requests);
	return result;

}

void* ListenSocket(void* arg) {
    int listen_fd = (int)(uintptr_t)arg;
    struct sockaddr_un socket_address;
    int client_fd;
    FileStateInfo file_info;
    socklen_t address_length = sizeof(socket_address);

    if (listen(listen_fd, 5) < 0) {
        perror("server: listen");
        return NULL;
    }

    struct pollfd fds[1];
    fds[0].fd = listen_fd;
    fds[0].events = POLLIN;

    while (1) {
    	fflush(stdout);
    	int ready = poll(fds, 1, 10000);  // 5 секунд таймаут
    	printf("Exiting g_poll(), ready = %d, errno = %d\n", ready, errno);
    	fflush(stdout);
        if (ready == 0) {
            continue;  // Таймаут, продолжаем цикл
        }

    	if (ready == -1) {
    		if (errno == EINTR) continue;
    		printf("perror(poll)\n");
    		perror("poll");
    		break;
    	}

        if (fds[0].revents & POLLHUP) {
            printf("Connection closed\n");
            break;
        }

        if (fds[0].revents & G_IO_IN) {
            client_fd = accept(listen_fd, (struct sockaddr*)&socket_address, &address_length);
            if (client_fd < 0) {
                if (errno == EINTR) continue;
                perror("server: accept");
                continue;
            }

        	unsigned char buffer[1024];
        	ssize_t bytes_read = recv(client_fd, buffer, sizeof(buffer), 0);
        	close(client_fd);

        	if (bytes_read <= 0) continue;

        	// Парсим данные
        	buffer[bytes_read] = '\0';
        	file_info = ParseFileInfo(buffer, bytes_read);
        	if (!file_info.uri) {
        		continue;  // пропускаем некорректные сообщения
        	}

            printf("Received: uri - %s, index - %i\n", file_info.uri, file_info.icon_index);

            CajaFileInfo* file = caja_file_info_lookup_for_uri(file_info.uri);
            if (file != NULL) {
            	g_hash_table_insert(pending_updates, g_strdup(file_info.uri), GINT_TO_POINTER(file_info.icon_index));
            	printf(">>> hash add new icon\n");
                caja_file_info_invalidate_extension_info(file);
                g_object_unref(file);  // Не забудьте освободить объект CajaFileInfo
            } else {
                printf("Failed to lookup file info for URI: %s\n", file_info.uri);
            }

        	g_free(file_info.uri);

        }
    }

    printf("ListenSocket ending\n");
    return NULL;
}

void FileStateListener() {
	struct sockaddr_un socket_address;
	int state_listener_socket;

	printf(">>get socket descriptor\n");
	state_listener_socket = socket(AF_UNIX, SOCK_STREAM, 0);
	if (state_listener_socket < 0) {
		perror("client: socket");
		exit(1);
	}

	socket_address.sun_family = AF_UNIX;
	strncpy(socket_address.sun_path, ADDRESS_FOR_NOTIF_STATE, sizeof(socket_address.sun_path) - 1);
	socket_address.sun_path[sizeof(socket_address.sun_path) - 1] = '\0';

	unlink(ADDRESS_FOR_NOTIF_STATE);
	const int length = sizeof(socket_address);
	if (bind(state_listener_socket, (struct sockaddr*)&socket_address, length) < 0) {
		perror("server: bind");
		exit(1);
	}

	if (!pending_updates) {
		pending_updates = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	}

	printf(">>listening to requests\n");
	pthread_t th1;
	pthread_create(&th1, NULL, ListenSocket, (void*)(intptr_t)state_listener_socket);
	printf(">>listener created\n");
}

int ChangeFileEmblem(CajaFileInfo* file, const int icon_index) {
	switch (icon_index) {
	case 1: {
		caja_file_info_add_emblem(file, emblems[1]);
		printf("emblem: 1 \n");
		break;
	}
	case 2: {
		caja_file_info_add_emblem(file, emblems[2]);
		printf("emblem: 2 \n");
		break;
	}
	case 3: {
		caja_file_info_add_emblem(file, emblems[3]);
		printf("emblem: 3 \n");
		break;
	}
	case 4: {
		caja_file_info_add_emblem(file, emblems[4]);
		printf("emblem: 4 \n");
		break;
	}
	case 5: {
		caja_file_info_add_emblem(file, emblems[5]);
		printf("emblem: 5 \n");
		break;
	}
	case 8: {
		caja_file_info_add_emblem(file, emblems[8]);
		printf("emblem: 8 \n");
		break;
	}
	}

	caja_file_info_invalidate_extension_info(file);
	return 0;
}

static CajaOperationResult caja_3dstorage_extension_update_file_info(
	CajaInfoProvider* const info_provider,
	CajaFileInfo* const caja_file,
	GClosure* const update_complete,
	CajaOperationHandle** const operation_handle
) {
	setlocale(LC_ALL, "");
	GFile* location = caja_file_info_get_location(caja_file);
	char *uri = g_file_get_uri(location);
	char *path = g_file_get_path(location);

	if (!uri || path == NULL) {
	g_free(uri);
		g_free(path);
		g_object_unref(location);
		return CAJA_OPERATION_COMPLETE;
	}
	const char* strt = strstr(path, "3D-Storage");
	if (strt == NULL){
		g_free(uri);
		g_free(path);
		g_object_unref(location);
		return CAJA_OPERATION_COMPLETE;
	}

	if (pending_updates) {
		printf("\nPending updates keys:\n");
		GList *keys = g_hash_table_get_keys(pending_updates);
		for (GList *l = keys; l != NULL; l = l->next) {
			printf("Key: %s\n", (char*)l->data);
		}
		g_list_free(keys);
		printf("Current uri: %s\n", uri);
		gpointer icon_ptr = g_hash_table_lookup(pending_updates, uri);
		if (icon_ptr) {
			printf("\n>>>>>>Icon update from hash\n");
			int new_icon = GPOINTER_TO_INT(icon_ptr);
			ChangeFileEmblem(caja_file, new_icon);
			g_hash_table_remove(pending_updates, uri);
			printf("\n>>>>>>Icon update from hash end\n");
		}
		else {
			printf("\n>>>>>>Icon update from request start\n");
			ChangeFileEmblem(caja_file, RequestState(path));
		}
	}

	caja_file_info_invalidate_extension_info(caja_file);
	g_free(uri);
	g_free(path);
	g_object_unref(location);
	return CAJA_OPERATION_COMPLETE;
}


static void caja_3dstorage_extension_type_info_provider_iface_init(
	CajaInfoProviderIface* const iface) {
	iface->update_file_info = caja_3dstorage_extension_update_file_info;
}

static void caja_3dstorage_extension_class_init(
	GObjectClass* const caja_3dstorage_extension_class) {
	parent_class = g_type_class_peek_parent(caja_3dstorage_extension_class);
}


static void FreeArray(char** array, int itemCount) {
	for (int i = 0; i < itemCount; i++)
		free(array[i]);
	free(array);
}

static struct FilesPathToCommand GetFilePathsToCommand(GList* files) {
	int filesCountToCommand = g_list_length(files);
	int correctFilesCountToCommand = 0;

	if (filesCountToCommand == 0) {
		return (struct FilesPathToCommand){0, NULL};
	}

	char **pathCommand = malloc(filesCountToCommand * sizeof(char *));
	if (pathCommand == NULL)
		return (struct FilesPathToCommand){0, NULL};

	for (GList* l = files; l != NULL; l = l->next) {
		CajaFileInfo* file = CAJA_FILE_INFO(l->data);
		GFile *fp = caja_file_info_get_location(file);
		if (fp == NULL) {
			continue;
		}

		char *path = g_file_get_path(fp);
		if (path == NULL) {
			continue;
		}

		pathCommand[correctFilesCountToCommand] = strdup(path);
		if (pathCommand[correctFilesCountToCommand] == NULL) {
			FreeArray(pathCommand,correctFilesCountToCommand);
			g_free(path);
			return (struct FilesPathToCommand){0, NULL};
		}
		correctFilesCountToCommand++;
		g_free(path);
	}
	struct FilesPathToCommand result;
	result.count = correctFilesCountToCommand;
	result.Paths = pathCommand;
	return result;
}

static GList* caja_3dstorage_extension_get_file_items(CajaMenuProvider* provider,
	G_GNUC_UNUSED GtkWidget* window,
	GList* file_selection) {
	printf("caja_3dstorage_extension_get_file_items init\n");

	if (file_selection == NULL) {
		printf("file_selection return NULL\n");
		return NULL;
	}

	struct FilesPathToCommand filesPathToCommand = GetFilePathsToCommand(file_selection);
	if (filesPathToCommand.count == 0 || filesPathToCommand.Paths == NULL)
		return NULL;
	printf("Converted selected files count - %i \n", filesPathToCommand.count);

	DataContracts__CommandInvokeData ContextMenuRequestCommand = DATA_CONTRACTS__COMMAND_INVOKE_DATA__INIT;
	DataContracts__Guid ContextMenuRequestCommandGuid = DATA_CONTRACTS__GUID__INIT;

	ContextMenuRequestCommandGuid.lo = 5059794805538720836;
	ContextMenuRequestCommandGuid.hi = 5424190233400613762;
	ContextMenuRequestCommand.commandid = &ContextMenuRequestCommandGuid;
	ContextMenuRequestCommand.commandid->has_hi = 1;
	ContextMenuRequestCommand.commandid->has_lo = 1;
	ContextMenuRequestCommand.n_paths = filesPathToCommand.count;
	ContextMenuRequestCommand.paths= malloc(filesPathToCommand.count * sizeof(char *));
	if (ContextMenuRequestCommand.paths == NULL) {
		FreeArray(filesPathToCommand.Paths, filesPathToCommand.count);
		return NULL;
	}
	for (int i=0; i < filesPathToCommand.count; i++) {
		ContextMenuRequestCommand.paths[i] = calloc(strlen(filesPathToCommand.Paths[i]) + 1, sizeof(char));
		strcpy(ContextMenuRequestCommand.paths[i], filesPathToCommand.Paths[i]);
	}

	const DataContracts__CommandInvokeResult commandInvokeResult = SerializeAndSend(ContextMenuRequestCommand);
	GList *contextMenuItemslist = NULL;


	printf("ContextMenuRequestCommandResult is OK - %i\n",
		commandInvokeResult.result == DATA_CONTRACTS__SHELL_RESULT__Ok);
	printf("ContextMenuRequestCommandResult - HasData %i\n", commandInvokeResult.has_data == TRUE);
	if (commandInvokeResult.result == DATA_CONTRACTS__SHELL_RESULT__Ok)
		if (commandInvokeResult.has_data == TRUE) {
			const DataContracts__MenuData menu_data = *data_contracts__menu_data__unpack(
				NULL, commandInvokeResult.data.len, commandInvokeResult.data.data);
			printf("Client: Data received menu count %li\n", menu_data.n_items);

			if (menu_data.result== DATA_CONTRACTS__SHELL_RESULT__Ok && menu_data.n_items > 0)
				contextMenuItemslist = BuildContextMenu(provider, menu_data, file_selection, CommandInvoke);
		}
	FreeArray(filesPathToCommand.Paths, filesPathToCommand.count);
	return contextMenuItemslist;
}

static void CommandInvoke(CajaMenuItem* item) {
	GList* files = g_object_get_data(G_OBJECT(item), "3DStorage:files");

	if (files == NULL) {
		printf("return NULL\n");
		return;
	}

	struct FilesPathToCommand filesPathToCommand = GetFilePathsToCommand(files);
	if (filesPathToCommand.count == 0 || filesPathToCommand.Paths == NULL)
		return;

	DataContracts__CommandInvokeData dataContractsCommandData = DATA_CONTRACTS__COMMAND_INVOKE_DATA__INIT;
	DataContracts__Guid guid = DATA_CONTRACTS__GUID__INIT;

	guid.hi = (long)g_object_get_data(G_OBJECT(item), "commandIdHi");
	guid.lo = (long)g_object_get_data(G_OBJECT(item), "commandIdLo");

	dataContractsCommandData.commandid = &guid;
	dataContractsCommandData.commandid->has_hi = 1;
	dataContractsCommandData.commandid->has_lo = 1;
	dataContractsCommandData.n_paths = filesPathToCommand.count;
	dataContractsCommandData.paths = malloc(filesPathToCommand.count * sizeof(char*));
	if (dataContractsCommandData.paths == NULL) {
		FreeArray(filesPathToCommand.Paths, filesPathToCommand.count);
		return;
	}
	for (int i=0; i < filesPathToCommand.count; i++) {
		dataContractsCommandData.paths[i] = strdup(filesPathToCommand.Paths[i]);
		if (dataContractsCommandData.paths[i] == NULL) {
			for (int j = 0; j < i; j++) {
				free(dataContractsCommandData.paths[j]);
			}
			free(dataContractsCommandData.paths);
			FreeArray(filesPathToCommand.Paths, filesPathToCommand.count);
			return;
		}
	}

	printf(">>> commandInvoke hi - %lu\n", dataContractsCommandData.commandid->hi);
	printf(">>> commandInvoke lo - %lu\n", dataContractsCommandData.commandid->lo);
	SerializeAndSend(dataContractsCommandData);
	printf(">>> commandInvoke - done\n");
	FreeArray(filesPathToCommand.Paths, filesPathToCommand.count);
	FreeArray(dataContractsCommandData.paths,dataContractsCommandData.n_paths);
}

static void caja_3dstorage_extension_menu_provider_iface_init(
	CajaMenuProviderIface* const iface) {
	iface->get_file_items = caja_3dstorage_extension_get_file_items;
}


static void caja_register_types(
	GTypeModule* const module) {
	static const GTypeInfo info = {
			sizeof(GObjectClass),
			(GBaseInitFunc)NULL,
			(GBaseFinalizeFunc)NULL,
			(GClassInitFunc)caja_3dstorage_extension_class_init,
			(GClassFinalizeFunc)NULL,
			NULL,
			sizeof(GObject),
			0,
			(GInstanceInitFunc)NULL,
			(GTypeValueTable*)NULL
	};

	static const GInterfaceInfo type_info_provider_iface_info = {
			(GInterfaceInitFunc)caja_3dstorage_extension_type_info_provider_iface_init,
			(GInterfaceFinalizeFunc)NULL,
			NULL
	};

	static const GInterfaceInfo menu_provider_iface_info = {
			(GInterfaceInitFunc)caja_3dstorage_extension_menu_provider_iface_init,
			(GInterfaceFinalizeFunc)NULL,
			NULL
	};

	caja_3dstorage_extension_type = g_type_module_register_type(
		module,
		G_TYPE_OBJECT,
		"Caja3DStorageExtension",
		&info,
		0);

	g_type_module_add_interface(
		module,
		caja_3dstorage_extension_type,
		CAJA_TYPE_MENU_PROVIDER,
		&menu_provider_iface_info);

	g_type_module_add_interface(
		module,
		caja_3dstorage_extension_type,
		CAJA_TYPE_INFO_PROVIDER,
		&type_info_provider_iface_info);
}


GType caja_3dstorage_extension_get_type(void) {
	return caja_3dstorage_extension_type;
}


void caja_module_shutdown(void) {
	/*  Any module-specific shutdown  */
}


void caja_module_list_types(
	const GType** const types,
	int* const num_types) {
	*types = provider_types;
	*num_types = G_N_ELEMENTS(provider_types);
}


// Extension initialization
void caja_module_initialize(
	GTypeModule* const module) {
	printf(">>caja_module_initialize \n");
	printf(">>caja_module_version - %s\n", VERSION);
	printf(">>caja_module debug - null memory\n");
	caja_register_types(module);
	*provider_types = caja_3dstorage_extension_get_type();
	FileStateListener();
}

DataContracts__CommandInvokeResult SerializeAndSend(const DataContracts__CommandInvokeData instance) {
	int socket_for_requests = 0;
	int data_len = 0;
	struct sockaddr_un socket_address;
	DataContracts__CommandInvokeResult res = DATA_CONTRACTS__COMMAND_INVOKE_RESULT__INIT;
	res.result = DATA_CONTRACTS__SHELL_RESULT__Error;
	res.has_result = 1;
	res.has_data = 0;

	printf("\nSerializeAndSend - start \n");
	struct DataContractMessage serializedDataContractMessage = SerializeDataContractMessage(instance);

		if ((socket_for_requests = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
		printf("Client: Error on socket() call \n");
	}
	socket_address.sun_family = AF_UNIX;
	strcpy(socket_address.sun_path, ADDRESS_FOR_COMMAND);
	data_len = strlen(socket_address.sun_path) + sizeof(socket_address.sun_family);

	if (connect(socket_for_requests, (struct sockaddr*)&socket_address, data_len) == -1) {
		printf("Client: Error on connect call \n");
	}

	printf("Client: Connected \n");

	if (send(socket_for_requests, serializedDataContractMessage.message, serializedDataContractMessage.messageLength,
		0) == -1) {
		printf("Client: Error on send() size \n");
	}

	char recv_msg_length[5];
	if ((data_len = recv(socket_for_requests, recv_msg_length, 4, 0)) == 4) {
		const int sizeResMessage = BytesToInt(recv_msg_length);
		printf("Client: Data received size message: %i \n", sizeResMessage);
		char recv_msg[sizeResMessage + 1];
		if ((data_len = recv(socket_for_requests, recv_msg, sizeResMessage, 0)) == sizeResMessage) {
			printf("Client: Data received data_len 2: %i \n", data_len);
			close(socket_for_requests);
			res = *data_contracts__command_invoke_result__unpack(NULL, data_len, recv_msg);
		}
	}
	else {
		if (data_len < 0) {
			printf("Client: Error on recv() call \n");
		}
		else {
			printf("Client: Server socket closed \n");
			close(socket_for_requests);
		}
	}
	return res;
}

/*  EOF  */

