/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using Ascon.Pilot.SDK.Menu;
using System.ComponentModel.Composition;
using System.Linq;

namespace Ascon.Pilot.SDK.ShowReportSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    [ExportMetadata("Name", "Show report extension")]
    public class ShowReportExtension : IMenu<ObjectsViewContext>
    {
        private readonly IObjectsRepository _repository;
        private readonly ITabServiceProvider _tabServiceProvider;
        private const string MENU_ITEM_NAME = "showReportMenuItem";
        private const string REPORT_SUBMENU_ITEM_NAME = "showReportSubmenuItem";

        [ImportingConstructor]
        public ShowReportExtension(
            IObjectsRepository repository, 
            ITabServiceProvider tabServiceProvider)
        {
            _repository = repository;
            _tabServiceProvider = tabServiceProvider;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var selection = context.SelectedObjects.ToArray();
            if(!selection.Any())
                return;

            var commandIndex = builder.ItemNames.ToList().IndexOf("miSend") - 1;
            var index = commandIndex < 0 ? 0 : commandIndex + 1;

            var reportsSubmenuBuilder = builder.AddItem(MENU_ITEM_NAME, index)
                .WithHeader("Show report extension")
                .WithSubmenu();

            var reportItems = _repository.GetReportItems().ToList();
            for (var submenuItemIndex = 0; submenuItemIndex < reportItems.Count; submenuItemIndex++)
            {
                var reportItem = reportItems[submenuItemIndex];

                reportsSubmenuBuilder
                    .AddItem(REPORT_SUBMENU_ITEM_NAME + GetXamlName(reportItem.Id.ToString()), submenuItemIndex)
                    .WithHeader(reportItem.Name);
            }
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if(!name.StartsWith(REPORT_SUBMENU_ITEM_NAME))
                return;

            var reportItems = _repository.GetReportItems().ToList();
            var targetReport = reportItems.FirstOrDefault(x => name.EndsWith(GetXamlName(x.Id.ToString())));
            if (targetReport == null) 
                return;

            var selection = context.SelectedObjects.ToArray();

            _tabServiceProvider.ShowElement(
                targetReport.Id, 
                DateTime.MinValue,
                selection.Select(x => x.Id as object).ToArray());
        }

        private string GetXamlName(string value)
        {
            return new string(value.Where(char.IsLetterOrDigit).ToArray());
        }
    }
}
