/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows;
using System.Windows.Media;
using System.Xml.Serialization;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Toolbar;
using Ascon.Pilot.SDK.GraphicLayerSample;
using Ascon.Pilot.Theme.ColorScheme;


namespace Ascon.Pilot.SDK.XpsViewerSample
{
    [Export(typeof(IToolbar<XpsRenderContext>))]
    public class XpsRenderToolbarSample : IToolbar<XpsRenderContext>, IObserver<INotification>, IMouseLeftClickListener
    {
        private const string TbiAdd = "tbiAddGraphicLayerElement";
        private const string TbiZoom = "tbiZoomGraphicLayerElement";
        private const string TbiZoomToAnnotation = "tbiZoomToAnnotation";
        private const string TbiAddStamp = "tbiAddStamp";

        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;
        private readonly IXpsViewer _xpsViewer;
        private readonly List<Guid> _graphicLayerQueueForZoom;
        private IPerson _currentPerson;
        private IToolbarBuilder _builder;
        private int _currentFileIndex;
        private int _currentAnnotationFileIndex;

        private List<IFile> _objectFiles;

        private XpsRenderContext _context;

        private bool _tbiAddStampState = false;

        [ImportingConstructor]
        public XpsRenderToolbarSample(IObjectModifier modifier, IObjectsRepository repository, IXpsViewer xpsViewer, IPilotDialogService dialogService)
        {
            _modifier = modifier;
            _repository = repository;
            _xpsViewer = xpsViewer;
            var convertFromString = ColorConverter.ConvertFromString(dialogService.AccentColor);
            _objectFiles = new List<IFile>();
            _graphicLayerQueueForZoom = new List<Guid>();
            if (convertFromString != null)
            {
                var color = (Color)convertFromString;
                ColorScheme.Initialize(color, dialogService.Theme);
            }
            repository.SubscribeNotification(NotificationKind.ObjectGraphicLayerChanged).Subscribe(this);
        }

        public void Build(IToolbarBuilder builder, XpsRenderContext context)
        {
            _builder = builder;
            _context = context;

            var zoomToAnnotation = "Перейти к аннотации";
            var addGraphicElement = "Добавить графический элемент";
            var editGraphicElement = "Показать графический элемент";
            var addStamp = "Создать графический элемент в точке";

            if (!_builder.ItemNames.Contains(TbiZoomToAnnotation)
                && !_builder.ItemNames.Contains(TbiAdd) 
                && !_builder.ItemNames.Contains(TbiZoom)
                && !_builder.ItemNames.Contains(TbiAddStamp))
                _builder.AddSeparator(_builder.Count);

            AddOrReplaceButton(_builder, TbiZoomToAnnotation)
                .WithIcon(IconLoader.GetIcon("zoom_to_annotation_icon.svg"))
                .WithIsEnabled(context.IsDocumentLoaded)
                .WithHeader(zoomToAnnotation)
                .WithShowHeader(false)
                .WithHint(zoomToAnnotation);
            AddOrReplaceButton(_builder, TbiAdd)
                .WithIcon(IconLoader.GetIcon("add_graphic_layer.svg"))
                .WithIsEnabled(context.IsDocumentLoaded)
                .WithHeader(addGraphicElement)
                .WithShowHeader(false)
                .WithHint(addGraphicElement);

            AddOrReplaceButton(_builder, TbiZoom)
                .WithIcon(IconLoader.GetIcon("zoom_graphic_layer.svg"))
                .WithIsEnabled(context.IsDocumentLoaded)
                .WithHeader(editGraphicElement)
                .WithShowHeader(false)
                .WithHint(editGraphicElement);

            AddOrReplaceToggleButtonItem(_builder, TbiAddStamp, _tbiAddStampState)
                .WithIcon(IconLoader.GetIcon("add_stamp.svg"))
                .WithIsEnabled(context.IsDocumentLoaded)
                .WithHeader(addStamp)
                .WithShowHeader(false)
                .WithHint(addStamp);  
        }

        private IToolbarButtonItemBuilder AddOrReplaceButton(IToolbarBuilder builder, string name)
        {
            return builder.ItemNames.Contains(name) ? builder.ReplaceButtonItem(name) : builder.AddButtonItem(name, builder.Count);
        }

        private IToolbarButtonItemBuilder AddOrReplaceToggleButtonItem(IToolbarBuilder builder, string name, bool state)
        {
            return builder.ItemNames.Contains(name) ? builder.ReplaceToggleButtonItem(name).WithIsChecked(state) : builder.AddToggleButtonItem(name, builder.Count).WithIsChecked(state);
        }

        public void OnLeftMouseButtonClick(XpsRenderClickPointContext pointContext)
        {
            CreateGraphicLayerElementAtPoint(new MouseClickContext(pointContext.DataObject, pointContext.ClickPoint, pointContext.PageNumber));
            _xpsViewer.UnsubscribeLeftMouseClick(this);
            _tbiAddStampState = false; 
        }

        public void OnToolbarItemClick(string name, XpsRenderContext context)
        {
            _currentPerson = _repository.GetCurrentPerson();
            if (name == TbiZoomToAnnotation)
            {
                var annotations = GetAnnotationFiles(context);
                if (!annotations.Any())
                    return;

                if (_currentAnnotationFileIndex > annotations.Count - 1)
                    _currentAnnotationFileIndex = 0;
                var annotationFile = annotations.ElementAt(_currentAnnotationFileIndex);
                if (annotationFile != null)
                {
                    var fileName = annotationFile.Name;
                    var match = fileName.Split('_');
                    Guid guid;
                    if (match.Length > 1 && !string.IsNullOrEmpty(match[1]) && Guid.TryParse(match[1], out guid))
                        _xpsViewer.ZoomToElement(guid, 0.7);
                }
                _currentAnnotationFileIndex++;
                
            }

            if (name == TbiAdd)
                AddGraphicLayerTextElementAndZoom(context.DataObject);

            if (name == TbiZoom)
            {
                var graphicFiles = GetGraphicFilesFromCurrentVersion(context);
                if (!graphicFiles.Any())
                    return;
                    
                if (_currentFileIndex > graphicFiles.Count - 1)
                    _currentFileIndex = 0;
                var currentGraphicLayerFile = graphicFiles.ElementAt(_currentFileIndex);
                if (currentGraphicLayerFile != null)
                {
                    var match = Regex.Match(currentGraphicLayerFile.Name, $"(?<={GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT}).*?(?=_)");
                    Guid guid;
                    if (match.Success && Guid.TryParse(match.Value, out guid))
                        _xpsViewer.ZoomToElement(guid, 0.7);
                }
                _currentFileIndex++;
            }

            if (name == TbiAddStamp)
            {
                _xpsViewer.UnsubscribeLeftMouseClick(this);

                _xpsViewer.SubscribeLeftMouseClick(this);
                _tbiAddStampState = true;
                AddOrReplaceToggleButtonItem(_builder, TbiAddStamp, _tbiAddStampState);
            }
        }

        private List<IFile> GetGraphicFilesFromCurrentVersion(XpsRenderContext context)
        {
            var graphicFiles = GetFilesFromCurrentVersion(context).Where(f => f.Name.Contains(GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT)).ToList();
            return graphicFiles;
        }

        private List<IFile> GetAnnotationFiles(XpsRenderContext context)
        {
            var graphicFiles = GetFilesFromCurrentVersion(context).Where(f => f.Name.Contains("Annotation")).ToList();
            return graphicFiles;
        }

        private IEnumerable<IFile> GetFilesFromCurrentVersion(XpsRenderContext context)
        {
            return context.SelectedVersion == context.DataObject.ActualFileSnapshot.Created ?
                context.DataObject.ActualFileSnapshot.Files :
                context.DataObject.PreviousFileSnapshots.First(s => s.Created == context.SelectedVersion).Files;
        }

        private void AddGraphicLayerTextElementAndZoom(IDataObject dataObject)
        {
            var elementId = Guid.NewGuid();
            var element = GraphicLayerElementCreator.CreateStampWithDateTime().ToString();
            SaveToDataBaseXaml(dataObject, element, elementId);
            _graphicLayerQueueForZoom.Add(elementId);
        }

        private void SaveToDataBaseXaml(IDataObject dataObject, string xamlObject, Guid elementId)
        {
            var builder = _modifier.Edit(dataObject);
            var positionId = _currentPerson.MainPosition.Position;
            var name = GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT + elementId;
            var scale = new Point(1, 1);
            var element = GraphicLayerElementCreator.Create(0, 0, scale, 0, positionId, VerticalAlignment.Top, HorizontalAlignment.Left, 
                GraphicLayerElementConstants.XAML, elementId, _xpsViewer.CurrentPageNumber, true);

            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(GraphicLayerElement));
                serializer.Serialize(stream, element);
                stream.Position = 0;
                builder.AddFile(name, stream, DateTime.Now, DateTime.Now, DateTime.Now);
            }

            using (var textBlocksStream = new MemoryStream())
            {
                using (var writer = new StreamWriter(textBlocksStream))
                {
                    writer.Write(xamlObject);
                    writer.Flush();
                    builder.AddFile(GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT_CONTENT + element.ContentId, textBlocksStream, DateTime.Now, DateTime.Now, DateTime.Now);
                }
            }
            _modifier.Apply();
        }

        public async void OnNext(INotification notification)
        {
            _currentPerson = _repository.GetCurrentPerson();
            if (notification.ChangeKind == NotificationKind.ObjectGraphicLayerChanged && _currentPerson.Id == notification.UserId)
            {
                var loader = new ObjectLoader(_repository);
                var obj = await loader.Load(notification.ObjectId);
                
                var newFiles = obj.Files.ToList();
                var addedFileNames = newFiles.Select(f => f.Name).Except(_objectFiles.Select(f => f.Name)).ToList();
                foreach (var elementId in _graphicLayerQueueForZoom.ToList())
                {
                    if (addedFileNames.Any(name => name.StartsWith(GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT + elementId)))
                    {
                        _xpsViewer.ZoomToElement(elementId, 0.7);
                        _graphicLayerQueueForZoom.Remove(elementId);
                    }
                }
                _objectFiles = newFiles;
            }
        }

        private void CreateGraphicLayerElementAtPoint(MouseClickContext context)
        {
            var elementId = Guid.NewGuid();
            var positionId = _currentPerson.MainPosition.Position;
            var xamlContent = GraphicLayerElementCreator.CreateStampWithDateTime().ToString();
            var elementName = GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT + elementId + "_" + positionId;
            var element = CreateGraphicLayerElement(context.ClickPoint, context.PageNumber, elementId, positionId);

            var builder = _modifier.Edit(context.DataObject);
            using (var graphicElementStream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(GraphicLayerElement));
                serializer.Serialize(graphicElementStream, element);
                graphicElementStream.Position = 0;
                builder.AddFile(elementName, graphicElementStream, DateTime.Now, DateTime.Now, DateTime.Now);
            }

            using (var contentStream = new MemoryStream())
            {
                using (var writer = new StreamWriter(contentStream))
                {
                    writer.Write(xamlContent);
                    writer.Flush();
                    builder.AddFile(GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT_CONTENT + element.ContentId,
                        contentStream, DateTime.Now, DateTime.Now, DateTime.Now);
                }
            }
            _modifier.Apply();
        }

        private GraphicLayerElement CreateGraphicLayerElement(Point clickPoint, int pageNumber, Guid elementId, int positionId)
        {
            var scale = new Point(1, 1);
            return new GraphicLayerElement(elementId, Guid.NewGuid(), clickPoint.X, clickPoint.Y, positionId, scale,
                0, VerticalAlignment.Top, HorizontalAlignment.Left, GraphicLayerElementConstants.XAML, pageNumber, true);
        }

        public void OnError(Exception error) { }
        public void OnCompleted() { }
    }

    public class MouseClickContext
    {
        public Point ClickPoint { get; }
        public int PageNumber { get; }
        public IDataObject DataObject { get; }
        public MouseClickContext(IDataObject dataObject, Point clickPoint, int pageNumber)
        {
            DataObject = dataObject;
            PageNumber = pageNumber;
            ClickPoint = clickPoint;
        }
    }
}
