﻿using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Xml.Serialization;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Data;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace AttributeHistoryBot
{
    [Export(typeof(IObjectChangeProcessor))]
    [Export(typeof(ISettingsFeature2))]
    public class AttributeHistoryBot : IObjectChangeProcessor, ISettingsFeature2, IObserver<KeyValuePair<string, string>>
    {
        private readonly IObjectsRepository _repository;
        private readonly IMessageBuilder _messageBuilder;
        private readonly IMessagesRepository _messagesRepository;
        private readonly IAttributeFormatParser _attributeFormatParser;
        private readonly Dictionary<int, List<IAttribute>> _attributesByTypeId = new Dictionary<int, List<IAttribute>>();

        [ImportingConstructor]
        public AttributeHistoryBot(
            IObjectsRepository repository, 
            IPersonalSettings personalSettings, 
            IMessageBuilder messageBuilder, 
            IMessagesRepository messagesRepository,
            IAttributeFormatParser attributeFormatParser)
        {
            _repository = repository;
            _messageBuilder = messageBuilder;
            _messagesRepository = messagesRepository;
            _attributeFormatParser = attributeFormatParser;
            personalSettings.SubscribeSetting(Key).Subscribe(this);
            UpdateSettings(personalSettings.GetPersonalSettingValue(Key));
        }

        public bool ProcessChanges(IEnumerable<DataObjectChange> changes, IObjectModifier modifier)
        {
            foreach (var change in changes)
            {
                if(change.Old == null || !_attributesByTypeId.TryGetValue(change.New.Type.Id, out var attributesToJournal))
                    continue;

                var stringBuilder = new StringBuilder();
                foreach (var attribute in attributesToJournal)
                {
                    change.Old.Attributes.TryGetValue(attribute.Name, out var oldValue);
                    change.New.Attributes.TryGetValue(attribute.Name, out var newValue);
                    var diff = AttrDiffProvider.CalcDiff(oldValue, newValue, attribute, _attributeFormatParser, _repository);
                    if (diff != null)
                    {
                        var oldValueText = diff.OldValue != null ? $" c {diff.OldValue}" : null;
                        stringBuilder.AppendLine($"Изменил(а) {attribute.Title}{oldValueText} на {diff.NewValue}");
                    }
                }

                if (stringBuilder.Length > 0)
                {
                    _ = AddSystemMessageToObjectChat(change.New, stringBuilder.ToString());
                }
            }
            return true;
        }

        private async Task AddSystemMessageToObjectChat(IDataObject obj, string messageText)
        {
            var chat = await _messagesRepository.LoadChatAsync(obj.Id);
            if (chat.LastMessageId == null)
            {
                var creation = _messageBuilder.CreateMessage(MessageType.ChatCreation, obj.Id)
                    .AddChatCreationData(obj.DisplayName, string.Empty, ChatKind.ObjectRelated);
                await _messagesRepository.SendMessageAsync(creation.Message);

                var relation = _messageBuilder.CreateMessage(MessageType.ChatRelation, obj.Id).
                    AddChatRelation(ChatRelationType.Relation, obj.Id);

                await _messagesRepository.SendMessageAsync(relation.Message);
            }

            var message = _messageBuilder.CreateMessage(MessageType.TextMessage, obj.Id);
            message.AddText(messageText, true);
            await _messagesRepository.SendMessageAsync(message.Message);
        }

        public void SetValueProvider(ISettingValueProvider settingValueProvider)
        {
        }

        public string Key => "AttributeHistoryBot-8D68B8C6-AD18-4F10-A964-CB737249A049";
        public string Title => "Настройки журналирования изменения атрибутов";
        public FrameworkElement Editor => null;
        public bool IsValid(string settingsItemValue)
        {
            return string.IsNullOrEmpty(settingsItemValue) || RulesSerializer.IsValid(settingsItemValue);
        }

        private void UpdateSettings(string value)
        {
            _attributesByTypeId.Clear();
            var rules = RulesSerializer.Deserialize(value);
            foreach (var rule in rules)
            {
                var type = _repository.GetType(rule.Type);
                if (type != null && type.Id != -1 && rule.Attributes.Any())
                    _attributesByTypeId[type.Id] = type.Attributes.Where(x => rule.Attributes.Contains(x.Name)).ToList();
            }
        }

        void IObserver<KeyValuePair<string, string>>.OnNext(KeyValuePair<string, string> value)
        {
            if(value.Key != Key)
                return;
            UpdateSettings(value.Value);
        }

        void IObserver<KeyValuePair<string, string>>.OnError(Exception error)
        {
        }

        void IObserver<KeyValuePair<string, string>>.OnCompleted()
        {
        }
    }

    public class TypeInfo
    {
        [XmlAttribute("Name")]
        public string TypeName { get; set; }

        [XmlArray("Attributes")]
        [XmlArrayItem("Attribute")]
        public List<string> Attributes { get; set; } = new List<string>();
    }
}
