﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Text;
using System.Windows;
using Ascon.Pilot.SDK;
using PdfSharp.Drawing;
using PdfStamper.Extensions;
using PdfStamper.Models;
using PdfStamper.Serializing;
using PdfStamper.ViewModels;

namespace PdfStamper
{
    [Export(typeof(IPdfStamper))]
    [Export(typeof(ISettingsFeature2))]
    public class PdfStamper : IPdfStamper, ISettingsFeature2
    {
        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;
        private readonly IPersonalSettings _personalSettings;
        private readonly IObjectsRepository _repository;
        private ISettingValueProvider _settingValueProvider;
        private readonly CertificateReader _certificateReader;
        private readonly XSize _stampSize = new XSize(XUnit.FromMillimeter(77).Point, XUnit.FromMillimeter(25).Point);
        private SettingsViewModel _settingsViewModel;

        [ImportingConstructor]
        public PdfStamper(IObjectModifier modifier, IFileProvider fileProvider, IPersonalSettings personalSettings, IObjectsRepository repository)
        {
            _modifier = modifier;
            _fileProvider = fileProvider;
            _personalSettings = personalSettings;
            _repository = repository;
            _certificateReader = new CertificateReader(fileProvider);
        }

        public string Key => Constants.STAMP_FEATURE_KEY;

        public string Title => Properties.Resources.StampSettingsFeatureName;

        public FrameworkElement Editor
        {
            get
            {
                _settingsViewModel?.Dispose();
                _settingsViewModel = new SettingsViewModel(_settingValueProvider, _repository);
                return new SettingsView { DataContext = _settingsViewModel };
            }
        }

        public bool IsValid(string settingsItemValue)
        {
            var settings = Serializer.DeserializeSettings(settingsItemValue);

            if (settings.Count == 0)
                return false;

            var errorsBuilder = new StringBuilder();
            foreach (var set in settings)
            {
                foreach (var baseModelValue in set.Value)
                {
                    if (!baseModelValue.IsValid(out var errors))
                        errorsBuilder.AppendLine($"Errors in settings for type {set.Key}: {errors}");
                }
            }

            if (errorsBuilder.Length > 0)
            {
                MessageBox.Show(errorsBuilder.ToString(), "Errors in settings", MessageBoxButton.OK);
                return false;
            }
            return true;
        }

        public void SetValueProvider(ISettingValueProvider settingValueProvider)
        {
            _settingValueProvider = settingValueProvider;
        }

        public bool CanAddStamps(PdfDocumentContext context)
        {
            if (context.Document == null)
                return false;

            var signatureRequests = _certificateReader.GetSignatureRequests(context.Document, context.Version);
            if (!signatureRequests.Any())
                return false;

            var settings = GetSettings();
            if (!settings.TryGetValue(context.Document.Type.Name, out var typeSetting))
                return false;

            return settings.Any() && typeSetting.Any(bM => bM.PutStamp);
        }

        private Dictionary<string, List<BaseSettingModel>> GetSettings()
        {
            var settings = _personalSettings.GetCommonSettingValue(Constants.STAMP_FEATURE_KEY).FirstOrDefault(x => x != null);
            return Serializer.DeserializeSettings(settings);
        }

        public void AddStamps(Stream stream, PdfStamperMode mode, PdfDocumentContext context)
        {
            if (context.Document == null)
                return;

            var certInfos = _certificateReader.GetInfosForStamps(context.Document, context.Version).ToList();
            if (!certInfos.Any())
                return;

            var settings = GetSettings();
            if (!settings.Any())
                return;

            var docTypeSetting = settings.FirstOrDefault(x => x.Key.Equals(context.Document.Type.Name));

            using (var stampCreator = new StampCreator(stream, mode))
            {
                StampPosition signStampPosition = null;
                StampPosition toProductionPosition = null;
                foreach (var sInfo in certInfos)
                {
                    var toProductionSettingModel = docTypeSetting.Value.First(x => x.SettingType == SettingType.IntoProduction) as IntoProductionSettingModel;
                    if (toProductionSettingModel != null
                        && !string.IsNullOrEmpty(sInfo.SignerRole)
                        && sInfo.SignerRole.Equals(toProductionSettingModel.ProductionRole, StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (!toProductionSettingModel.PutStamp)
                            continue;

                        InjectIntoProductionStamp(context, stampCreator, toProductionSettingModel, sInfo, ref toProductionPosition);
                        continue;
                    }

                    var signatureSettingModel = docTypeSetting.Value.First(x => x.SettingType == SettingType.Signature) as SignatureSettingModel;
                    if (signatureSettingModel != null && !signatureSettingModel.PutStamp)
                        continue;

                    InjectSignatureStamp(context, stampCreator, signatureSettingModel, sInfo, ref signStampPosition);
                }

                stampCreator.SaveStamps(stream);
            }
        }

        private void InjectIntoProductionStamp(PdfDocumentContext context, StampCreator stampCreator, BaseSettingModel baseSettingModel, CertificateInfo sInfo, ref StampPosition toProductionStampPosition)
        {
            var pageNumber = GetPageNumberForStamp(baseSettingModel, stampCreator);
            if (pageNumber < 0)
                return;

            var existingStampPosition = GetExistingStampDataFile(context, sInfo.Id);

            if (existingStampPosition == null && toProductionStampPosition == null)
                toProductionStampPosition = CalculateFirstStampPositionFromSettings(baseSettingModel, stampCreator.GetPageSize(pageNumber));

            stampCreator.AddToProductionStamp(pageNumber, sInfo, existingStampPosition ?? toProductionStampPosition, _stampSize);

            if (existingStampPosition != null)
                return;

            var xOffSetPrevToProductionStamp = XUnit.FromMillimeter(baseSettingModel.OffsetPreviousX);
            var yOffSetPrevToProductionStamp = XUnit.FromMillimeter(baseSettingModel.OffsetPreviousY);
            toProductionStampPosition.X += xOffSetPrevToProductionStamp.Point;
            toProductionStampPosition.Y += yOffSetPrevToProductionStamp.Point;
        }

        private void InjectSignatureStamp(PdfDocumentContext context, StampCreator stampCreator, SignatureSettingModel signatureSettingModel, CertificateInfo sInfo, ref StampPosition signatureStampPosition)
        {
            var pageNumber = GetPageNumberForStamp(signatureSettingModel, stampCreator);
            if (pageNumber < 0)
                return;

            var existingStampPosition = GetExistingStampDataFile(context, sInfo.Id);

            if (existingStampPosition == null && signatureStampPosition == null)
                signatureStampPosition = CalculateFirstStampPositionFromSettings(signatureSettingModel, stampCreator.GetPageSize(pageNumber));

            stampCreator.AddSignatureStamp(pageNumber, sInfo, existingStampPosition ?? signatureStampPosition, _stampSize, signatureSettingModel);

            if (existingStampPosition != null)
                return;

            var xOffSetPrevSignStamp = XUnit.FromMillimeter(signatureSettingModel.OffsetPreviousX);
            var yOffSetPrevSignStamp = XUnit.FromMillimeter(signatureSettingModel.OffsetPreviousY);
            signatureStampPosition.X += xOffSetPrevSignStamp.Point;
            signatureStampPosition.Y += yOffSetPrevSignStamp.Point;
        }

        private static int GetPageNumberForStamp(BaseSettingModel setting, StampCreator stampCreator)
        {
            var pagesCount = stampCreator.GetPagesCount();
            if (!setting.IsOnLastPage && pagesCount < setting.PageNumber)
                return -1;

            var pageNumber = setting.IsOnLastPage
                ? pagesCount - 1
                : setting.PageNumber - 1;

            return pageNumber;
        }

        public void OnStampPositionChanged(StampPositionArgs args)
        {
            var snapShot = Tools.GetFilesSnapshot(args.Context.Document, args.Context.Version);
            var currentFile = snapShot.Files.FirstOrDefault(n => n.Name.Equals($"{args.StampId}{Constants.SIGNATURE_STAMP_EXT}"));

            var builder = _modifier.Edit(args.Context.Document);
            var stream = Tools.GetStreamFromStampData(new StampPosition { X = args.X, Y = args.Y });

            if (currentFile != null)
                builder.ReplaceFileInSnapshot(snapShot.Created, currentFile.Id, currentFile.Name, stream, currentFile.Created, DateTime.Now, DateTime.Now);
            else
                builder.AddFileInSnapshot(args.Context.Version, $"{args.StampId}{Constants.SIGNATURE_STAMP_EXT}", stream, DateTime.Now, DateTime.Now, DateTime.Now, out _);

            _modifier.Apply();
        }

        private static StampPosition CalculateFirstStampPositionFromSettings(BaseSettingModel baseSettingModel, XSize pageSize)
        {
            var xUnitOffset = XUnit.FromMillimeter(baseSettingModel.OffsetX);
            var yUnitOffSet = XUnit.FromMillimeter(baseSettingModel.OffsetY);

            switch (baseSettingModel.Position)
            {
                case Position.LeftTop:
                    return new StampPosition { X = xUnitOffset.Point, Y = yUnitOffSet.Point };

                case Position.RightTop:
                    return new StampPosition { X = pageSize.Width - xUnitOffset.Point, Y = yUnitOffSet.Point };

                case Position.LeftBottom:
                    return new StampPosition { X = xUnitOffset.Point, Y = pageSize.Height - yUnitOffSet.Point };

                case Position.RightBottom:
                    return new StampPosition { X = pageSize.Width - xUnitOffset.Point, Y = pageSize.Height - yUnitOffSet.Point };
                default:
                    throw new ArgumentOutOfRangeException(nameof(baseSettingModel.Position));
            }
        }

        private StampPosition GetExistingStampDataFile(PdfDocumentContext context, string stampId)
        {
            var snapShot = Tools.GetFilesSnapshot(context.Document, context.Version);
            var currentFile = snapShot.Files.FirstOrDefault(n => n.Name.Equals($"{stampId}{Constants.SIGNATURE_STAMP_EXT}"));
            return currentFile == null ? null : Serializer.DeserializeFromStream(_fileProvider.OpenRead(currentFile));
        }
    }
}