/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;
using Pilot.Xps.Domain.Context;
using Pilot.Xps.Entities;

namespace Pilot.Xps.Domain.Signatures
{
    public interface IXpsDocumentSigner
    {
        void SignDocumentWithCertificate(INObject document, IEnumerable<int> positionIds, X509Certificate2 certificate);
        IList<XpsDigitalSignature> GetSignatures(INObject document);
        IList<XpsDigitalSignature> GetSignatures(INObject document, DateTime snapshotDateTime);
    }

    public class XpsDocumentSigner : IXpsDocumentSigner
    {
        private readonly IFileLoader _fileLoader;
        private readonly IFileStorageProvider _fileStorageProvider;
        private readonly IBackend _backend;

        public XpsDocumentSigner(IBackend backend, IFileLoader fileLoader, IFileStorageProvider fileStorageProvider)
        {
            _fileLoader = fileLoader;
            _fileStorageProvider = fileStorageProvider;
            _backend = backend;
        }

        public void SignDocumentWithCertificate(INObject document, IEnumerable<int> positionIds, X509Certificate2 certificate)
        {
            var file = document.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            var buffer = _fileLoader.Download(file);
            var positions = positionIds.ToList();

            using (var stream = new MemoryStream(buffer))
            using (var context = new DocumentContext(stream))
            {
                var modifier = new Modifier(_backend);
                var xpsPartLoader = new XpsPartLoader(document.ActualFileSnapshot.Files, _fileLoader);
                var signer = new SignatureBuilder(_backend, xpsPartLoader);
                var signatures = signer.BuildSignatureWithRequestForAllPositions(context, document, positions, certificate);

                var builder = modifier.EditObject(document.Id);
                foreach (var signature in signatures)
                {
                    var signatureStream = signature.Serialize();
                    var now = DateTime.UtcNow;
                    var documentInfo = new DocumentInfo(Constants.DIGITAL_SIGNATURE, () => signatureStream, now, now, now);
                    var signatureFile = builder.AddFile(documentInfo, _fileStorageProvider);
                    builder.SetSign(signature.SpotId, signatureFile.Id.ToString());
                }

                if (modifier.AnyChanges())
                    modifier.Apply();
            }
        }

        public IList<XpsDigitalSignature> GetSignatures(INObject document)
        {
            return GetXpsDigitalSignatures(document.ActualFileSnapshot.Files.ToList());
        }

        public IList<XpsDigitalSignature> GetSignatures(INObject document, DateTime snapshotDateTime)
        {
            var snapshots = document.PreviousFileSnapshots.Union(new [] { document.ActualFileSnapshot });
            var snapshot = snapshots.FirstOrDefault(s => s.Created == snapshotDateTime);
            return snapshot == null ? 
                new List<XpsDigitalSignature>() : 
                GetXpsDigitalSignatures(snapshot.Files.ToList());
        }

        private IList<XpsDigitalSignature> GetXpsDigitalSignatures(IList<INFile> files)
        {
            var file = files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            if (file == null)
                return new List<XpsDigitalSignature>();

            var buffer = _fileLoader.Download(file);
            using (var stream = new MemoryStream(buffer))
            using (var context = new DocumentContext(stream))
            {
                var xpsPartLoader = new XpsPartLoader(files, _fileLoader);
                var signer = new SignatureBuilder(_backend, xpsPartLoader);
                var signatures = signer.BuildSignatures(context, files);
                var result = signatures.Select(s => new XpsDigitalSignature
                {
                    Id = s.SignId ?? Guid.Empty,
                    SignDate = string.IsNullOrEmpty(s.SignDate) ? DateTime.MinValue : DateTime.Parse(s.SignDate),
                    Signer = s.SignerName,
                    IsCertificateValid = s.IsCertificateValid,
                    IsSigned = s.IsSigned,
                    IsValid = s.IsValid,
                    CanUserSign = s.CanUserSign,
                    IsAdditional = s.IsAdditional,
                    Role = s.Role,
                    RequestedSigner = s.RequestedSigner
                });

                return result.ToList();
            }
        }
    }
}
