/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Windows;
using System.Xml.Serialization;
using Ascon.Pilot.Common;

namespace Pilot.Xps.Domain
{
    public interface IGraphicLayerElementContent
    {
        FrameworkElement Element { get; }
        Stream GetStream();
    }

    [Serializable]
    public class GraphicLayerElement
    {
        public Guid ElementId { get; set; }
        public Guid ContentId { get; set; }
        public double OffsetY { get; set; }
        public double OffsetX { get; set; }
        public double Angle { get; set; }
        public int PositionId { get; set; }
        public int PageNumber { get; set; }
        public Point Scale { get; set; }
        public string ContentType { get; set; }
        public bool IsFloating { get; set; }
        public string MetaData { get; set; }
        [XmlIgnore]
        public IGraphicLayerElementContent GraphicLayerElementContent { get; set; }
        public VerticalAlignment VerticalAlignment { get; set; }
        public HorizontalAlignment HorizontalAlignment { get; set; }
        public string FileName { get; set; }
        public GraphicLayerElement() { }

        public GraphicLayerElement(Guid elementId, Guid contentId, double offsetY, double offsetX, int positionId, Point scale, double angle,
            VerticalAlignment verticalAlignment, HorizontalAlignment horizontalAlignment, string contentType, string metaData, int pageNumber, bool isFloating)
        {
            ElementId = elementId;
            ContentId = contentId;
            OffsetY = offsetY;
            OffsetX = offsetX;
            Scale = scale;
            Angle = angle;
            PositionId = positionId;
            VerticalAlignment = verticalAlignment;
            HorizontalAlignment = horizontalAlignment;
            ContentType = contentType;
            MetaData = metaData;
            PageNumber = pageNumber;
            IsFloating = isFloating;
        }

        public string GetFileName()
        {
            return string.IsNullOrEmpty(FileName) ? XpsConstantsProvider.GetGraphicLayerElementDefinition() + ElementId : FileName;
        }

        public string GetContentFileName()
        {
            return XpsConstantsProvider.GetGraphicLayerElementContent() + ContentId;
        }

        public Stream Serialize()
        {
            var serializer = new XmlSerializer(typeof(GraphicLayerElement));
            var elementStream = new MemoryStream();
            serializer.Serialize(elementStream, this);
            return elementStream;
        }

        public static GraphicLayerElement Deserialize(Stream stream)
        {
            var serializer = new XmlSerializer(typeof(GraphicLayerElement));
            var graphicElement = serializer.Deserialize(stream) as GraphicLayerElement;
            return graphicElement;
        }

        public void Actualize(GraphicLayerElement actualGraphicElement)
        {
            OffsetY = actualGraphicElement.OffsetY;
            OffsetX = actualGraphicElement.OffsetX;
            Scale = actualGraphicElement.Scale;
            Angle = actualGraphicElement.Angle;
            PositionId = actualGraphicElement.PositionId;
            VerticalAlignment = actualGraphicElement.VerticalAlignment;
            HorizontalAlignment = actualGraphicElement.HorizontalAlignment;
            ContentType = actualGraphicElement.ContentType;
            MetaData = actualGraphicElement.MetaData;
            PageNumber = actualGraphicElement.PageNumber;
        }
    }

    public enum GraphicLayerOption
    {
        /// <summary>Inject in regard of IsFloating's parameter state.</summary>
        InjectFloatingRelatable,
        /// <summary>Inject all gragraphic layers, text labels and barcode.</summary>
        ForceInject,
        /// <summary>Keep all gragraphic elements as floating.</summary>
        KeepAllAsFloating,
        /// <summary>Do not add anything to file.</summary>
        LoseGraphicLayers
    }
}
