/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Windows.Annotations;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.DataClasses;
using Pilot.Xps.Domain.Annotations;
using Pilot.Xps.Domain.Context;
using Pilot.Xps.Domain.GraphicLayer;
using Pilot.Xps.Domain.IO;
using Pilot.Xps.Domain.Labels;
using Pilot.Xps.Domain.Render;
using Pilot.Xps.Domain.Signatures;
using Pilot.Xps.Domain.Tools;

namespace Pilot.Xps.Domain
{
    public interface IXpsDocumentMergeService
    {
        byte[] MergeDocument(INObject document, INType type);
    }

    public class XpsDocumentMergeService : IXpsDocumentMergeService
    {
        private readonly IFileLoader _fileLoader;

        public XpsDocumentMergeService(IFileLoader fileLoader)
        {
            _fileLoader = fileLoader;
        }

        public byte[] MergeDocument(INObject document, INType type)
        {
            byte[] result = new byte[] { };
            ThreadExtensions.RunInStaThread<Stream>(() =>
            {
                var partLoader = new XpsPartLoader(document.ActualFileSnapshot.Files.ToList(), _fileLoader);
                var body = partLoader.DownloadXpsBody();
                using (var fileStream = new MemoryStream(body))
                using (var stream = IoUtils.CreateMemoryStream(fileStream))
                {
                    var signatures = partLoader.DownloadSignatures(out var definitions);
                    MergeSignatureDefinitions(definitions, stream);
                    MergeSignatures(signatures, stream);
                    var annotations = partLoader.DownloadAnnotations();
                    MergeAnnotations(annotations, stream);
                    var annotationMessages = partLoader.DownloadAnnotationsMessages();
                    MergeAnnotationsMessages(annotationMessages, stream);
                    var graphicLayers = partLoader.DownloadGraphicLayerElements();
                    MergeGraphicLayerElements(graphicLayers, stream, signatures.Any());
                    var textLabels = partLoader.DownloadTextLabels();
                    var barcode = partLoader.DownloadBarcode();
                    var resultStream = MergeLabelsAndBarcode(textLabels, barcode, signatures.Any(), document, type, stream);
                    result = resultStream.ReadFully();
                }
            });

            return result;
        }

        private void MergeSignatures(IList<DigitalSignature> signatures, Stream inStream)
        {
            if (signatures != null && signatures.Count > 0)
                XpsSignatureHelper.MergeDigitalSignatures(signatures, inStream);
        }

        private void MergeSignatureDefinitions(IList<XpsSignatureDefinition> signatureDefinitions, Stream inStream)
        {
            if (signatureDefinitions != null && signatureDefinitions.Count > 0)
                XpsSignatureHelper.AddDefinitions(signatureDefinitions, inStream);
        }

        private static void MergeAnnotations(IList<Annotation> annotations, Stream inStream)
        {
            if (annotations != null && annotations.Count > 0)
            {
                using (var package = Package.Open(inStream, FileMode.Open, FileAccess.ReadWrite))
                using (var store = new AnnotationsStore(package))
                {
                    foreach (var annotation in annotations.Where(annotation => store.GetAnnotation(annotation) == null))
                    {
                        store.AddAnnotation(annotation);
                    }
                }
            }
        }

        private void MergeAnnotationsMessages(AnnotationMessages annotationMessages, Stream inStream)
        {
            if (annotationMessages.Messages.Count > 0)
            {
                using (var package = Package.Open(inStream, FileMode.Open, FileAccess.ReadWrite))
                using (var messagesStream = XpsTools.GetAnnotationMessagesStreamInPackage(package, true))
                {
                    annotationMessages.Serialize(messagesStream);
                }
            }
        }

        private void MergeGraphicLayerElements(IList<GraphicLayerElement> graphicLayerElements, Stream inStream, bool hasSignatures)
        {
            var floatingGraphicElements = graphicLayerElements.Where(l => l.IsFloating).ToList();
            var embeddedGraphicElements = graphicLayerElements.Where(l => !l.IsFloating).ToList();

            if (floatingGraphicElements.Any())
            {
                var writer = new FloatingElementsReaderWriter();
                writer.WriteGraphicLayer(inStream, floatingGraphicElements);
            }

            if (embeddedGraphicElements.Any() && !hasSignatures)
            {
                var writer = new XpsWriter();
                using (var outStream = writer.InjectGraphicLayerElements(inStream, embeddedGraphicElements))
                {
                    IoUtils.CopyStream(outStream, inStream);
                }
            }
        }

        private Stream MergeLabelsAndBarcode(string textLabels, string barcode, bool hasSignatures, INObject document, INType type, Stream inStream)
        {
            var attributes = document.Attributes.ToDictionary(o => o.Key, o => o.Value.Value);
            var writer = new FloatingElementsReaderWriter();

            var floatingLabels = GetFloatingElements(textLabels, hasSignatures, GraphicLayerOption.InjectFloatingRelatable, XpsWriter.GetFloatingTextLabel);
            writer.WriteLabel(inStream, floatingLabels, attributes);
            var floatingBarcode = GetFloatingElements(barcode, hasSignatures, GraphicLayerOption.InjectFloatingRelatable, XpsWriter.GetFloatingBarcode);
            writer.WriteBarcode(inStream, floatingBarcode, attributes);

            var embeddedLabels = GetEmbeddedElements(textLabels, hasSignatures, GraphicLayerOption.InjectFloatingRelatable, XpsWriter.GetConstantTextLabel);
            var embeddedBarcode = GetEmbeddedElements(barcode, hasSignatures, GraphicLayerOption.InjectFloatingRelatable, XpsWriter.GetConstantBarcode);
            var typeAttributeNames = type.Attributes.Where(a => !a.IsService).Select(a => a.Name).ToList();
            return InjectBarcodeAndTextLabels(inStream, embeddedLabels, embeddedBarcode, attributes, typeAttributeNames);
        }

        private Stream InjectBarcodeAndTextLabels(Stream docStream, string textLabelConfiguration, string barcodeConfiguration, Dictionary<string, object> attributes, List<string> attributeNames)
        {
            var xpsWriter = new XpsWriter();
            var barcodeCreator = new BarcodeCreator();
            var merger = new BarcodeLabelsMerger(xpsWriter, barcodeCreator);
            return merger.Merge(docStream, textLabelConfiguration, barcodeConfiguration, attributes, attributeNames);
        }

        private T GetFloatingElements<T>(T elements, bool hasSignatures, GraphicLayerOption layerOption, Func<T, T> getFloating)
        {
            if (layerOption == GraphicLayerOption.LoseGraphicLayers)
                return default(T);

            if (layerOption.FloatingUnrelatable())
            {
                if (layerOption == GraphicLayerOption.ForceInject)
                    return default(T);
                if (layerOption == GraphicLayerOption.KeepAllAsFloating)
                    return elements;
            }

            if (hasSignatures)
                return elements;

            return getFloating(elements);
        }

        private T GetEmbeddedElements<T>(T elements, bool hasSignatures, GraphicLayerOption layerOption, Func<T, T> getEmbeded)
        {
            if (layerOption == GraphicLayerOption.LoseGraphicLayers)
                return default(T);

            if (layerOption.FloatingUnrelatable())
            {
                if (layerOption == GraphicLayerOption.ForceInject)
                    return elements;
                if (layerOption == GraphicLayerOption.KeepAllAsFloating)
                    return default(T);
            }

            if (hasSignatures)
                return default(T);

            return getEmbeded(elements);
        }
    }
}
