/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/
import { ICertificate, ISignatureVerificationResult } from '@pilotdev/pilot-web-sdk';
import { Observable, Subject } from 'rxjs';

export type QueueType = string | number | boolean | ICertificate [] | ISignatureVerificationResult;

class QueueItem<T> {
  constructor(public callback: () => Promise<T>, public emitter: Subject<T>) {
  }
}

export class Queue<T> {

  private _elements: QueueItem<T> [] = [];
  
  private _isConnected: boolean = false;

  enqueue(fn: () => Promise<T>): Observable<T> {
    const taskEmitter = new Subject<T>();
    this._elements.push({
      callback: fn,
      emitter: taskEmitter,
    });

    this.runProcessQueue();
    return taskEmitter.asObservable();
  }

  stop(): void {
    this.setConnected(false);
  }

  start(): void {
    this.setConnected(true);
  }

  private setConnected(isConnected: boolean): void {
    this._isConnected = isConnected;
    this.runProcessQueue();
  }

  private dequeue(): QueueItem<T> | undefined {
    return this._elements.shift();
  }

  private runProcessQueue(): void {
    setTimeout(() => {
      this.processQueue();
    }, 1);
  }

  private async processQueue(): Promise<void> {
    if (this._elements.length <= 0) return;

    if (this._isConnected) {
      await this.execute();
      this.runProcessQueue();
    }
  }

  private async execute(): Promise<void> {
    const item = this.dequeue();
    if (!item) 
      return;
    
    try {
      const response = await item.callback();
      item.emitter.next(response);
      item.emitter.complete();  
    } catch (error) {
      item.emitter.error(error);
      item.emitter.complete();
    }
  }
}
