﻿using System;
using System.Linq;
using System.ComponentModel;
using System.Collections.ObjectModel;
using System.Runtime.CompilerServices;
using Ascon.Pilot.Bim.SDK.Search;
using Ascon.Pilot.Bim.Search.SDK.Serialization;
using Ascon.Pilot.Bim.Search.SDK.ModelProperties;
using Ascon.Pilot.Bim.SDK.BottomPanelTabSample.Tools;
using Ascon.Pilot.Bim.SDK.BottomPanelTabSample.Models;

namespace Ascon.Pilot.Bim.SDK.BottomPanelTabSample
{
    public class SearchSetViewModel : ISearchSetExpression, INotifyPropertyChanged
    {
        private readonly IModelSearchService _searchService;
        private readonly ISearchController _searchController;
        private LogicalOperator _logicalOperator;
        private EqualityOperator _operator;
        private SearchExpressionPropertyKind _kind;
        private object _expression;
        private string _propertyName;
        
        public SearchSetViewModel(IModelSearchService searchService, ISearchController searchController)
        {
            _searchService = searchService;
            _searchController = searchController;
            PropertyNames = new ObservableCollection<string>(_searchService
                .GetSearchProperties().Select(x => $"{x.CategoryName}{Constants.CATEGORY_NAME_SPLITTER}{x.Name}").OrderBy(y => y));
        }

        public ObservableCollection<LogicalOperator> LogicalOperators { get; set; } = new ObservableCollection<LogicalOperator> { LogicalOperator.And, LogicalOperator.Or };
        public ObservableCollection<EqualityOperator> Operations { get; set; }
        public ObservableCollection<ModelPartSelection> Lookup { get; set; } = new ObservableCollection<ModelPartSelection>();
        public ObservableCollection<string> PropertyNames { get; set; }

        public SearchExpressionPropertyKind Kind
        {
            get => _kind;
            set
            {
                _kind = value;

                InvalidateOperations();
                OnPropertyChanged(nameof(Kind));
                _searchController.InvalidateSearch();
            }
        }

        public LogicalOperator LogicalOperation
        {
            get => _logicalOperator;
            set
            {
                _logicalOperator = value;
                OnPropertyChanged(nameof(LogicalOperation));
                _searchController.InvalidateSearch();
            }
        }

        public EqualityOperator Operator
        {
            get => _operator;
            set
            {
                _operator = value;
                InvalidateProperties();
                OnPropertyChanged(nameof(Operator));
            }
        }
        
        public object Expression
        {
            get => _expression;
            set
            {
                _expression = value;
                OnPropertyChanged(nameof(Expression));
                _searchController.InvalidateSearch();
            }
        }

        public string PropertyName
        {
            get => _propertyName;
            set
            {
                _propertyName = value;

                var dataType = _searchService
                    .GetSearchProperties()
                    .First(x => _propertyName.StartsWith(x.CategoryName, StringComparison.OrdinalIgnoreCase) && _propertyName.EndsWith(x.Name, StringComparison.OrdinalIgnoreCase)).DataType;
                switch (dataType)
                {
                    case PropertyDataType.Double:
                        Kind = SearchExpressionPropertyKind.Double;
                        break;
                    case PropertyDataType.DateTime:
                        Kind = SearchExpressionPropertyKind.DateTime;
                        break;
                    case PropertyDataType.Guid when _propertyName.Equals(ModelProperty.ModelPartId.ToString(), StringComparison.OrdinalIgnoreCase): 
                        Kind = SearchExpressionPropertyKind.Lookup;
                        Lookup = new ObservableCollection<ModelPartSelection>(_searchController.GetModelParts());
                        OnPropertyChanged(nameof(Lookup));
                        break;
                    default:
                        Kind = SearchExpressionPropertyKind.String;
                        break;
                }

                Expression = null;

                InvalidateProperties();
                OnPropertyChanged(nameof(PropertyName));
            }
        }

        private void InvalidateOperations()
        {
            Operations = Kind == SearchExpressionPropertyKind.Double 
                ? new ObservableCollection<EqualityOperator> { EqualityOperator.Defined, EqualityOperator.Equal, EqualityOperator.GreaterOrEqual, EqualityOperator.LessOrEqual } 
                : new ObservableCollection<EqualityOperator> { EqualityOperator.Defined, EqualityOperator.Equal };

            OnPropertyChanged(nameof(Operations));
        }


        private void InvalidateProperties()
        {
            OnPropertyChanged(nameof(IsString));
            OnPropertyChanged(nameof(IsLookup));
            OnPropertyChanged(nameof(IsDouble));
            _searchController.InvalidateSearch();
        }

        public bool IsDouble => Kind == SearchExpressionPropertyKind.Double && Operator != EqualityOperator.Defined;

        public bool IsLookup => Kind == SearchExpressionPropertyKind.Lookup && Operator != EqualityOperator.Defined;

        public bool IsString => !(IsDouble || IsLookup) && Operator != EqualityOperator.Defined;

        public bool IsValid()
        {
            if (Expression == null && Operator == EqualityOperator.Defined)
                return true;

            if (Expression == null)
                return false;

            switch (Kind)
            {
                case SearchExpressionPropertyKind.Double:
                    if (double.TryParse(Expression.ToString(), out _))
                        return true;
                    break;
                case SearchExpressionPropertyKind.DateTime:
                    if (DateTime.TryParse(Expression.ToString(), out _))
                        return true;
                    break;
                case SearchExpressionPropertyKind.Lookup:
                case SearchExpressionPropertyKind.String:
                    return true;
            }

            return false;
        }

        public override string ToString()
        {
            return "SearchSetViewModel";
        }

        #region NotifyPropertyChanged

        public event PropertyChangedEventHandler PropertyChanged;
        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }

        #endregion
    }
}
