﻿using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Globalization;
using System.Reactive.Linq;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.Collections.Generic;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.Search.SDK;
using Ascon.Pilot.Bim.Search.SDK.Builders;
using Ascon.Pilot.Bim.Search.SDK.ModelProperties;
using Ascon.Pilot.Bim.Search.SDK.Serialization;
using Ascon.Pilot.Bim.Search.SDK.SearchExpressions;
using NavisWorksExchangeType;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace Ascon.Pilot.Bim.SDK.ImportSearchSetsSample.Import
{
    public class ImportService
    {
        private readonly IObjectModifier _objectModifier;
        private readonly IObjectsRepository _objectsRepository;
        private readonly IType _searchSetType;
        private readonly IType _searchFolderType;
        private IDataObject _coordinationModel;
        private IDisposable _subscription;
        private readonly StringBuilder _parseErrors = new StringBuilder();

        private static readonly XmlSerializer ExchangeSerializer = new XmlSerializer(typeof(Exchange));

        public ImportService(Guid coordinationModelId, IObjectsRepository repository, IObjectModifier modifier)
        {
            _objectsRepository = repository;
            _objectModifier = modifier;
            _searchSetType = _objectsRepository.GetType(Bim.Search.SDK.TypeNames.SearchSet);
            _searchFolderType = _objectsRepository.GetType(Bim.Search.SDK.TypeNames.SearchSetFolder);
            SubscribeForModelUpdates(coordinationModelId, OnModelUpdated);
        }
        
        public void ImportFromNavisWorks()
        {
            var dialog = new Microsoft.Win32.OpenFileDialog
            {
                Title = "Choose xml \"Autodesk Navisworks\" file for import",
                DefaultExt = ".xml",
                Filter = "Xml files (.xml)|*.xml"
            };

            if (dialog.ShowDialog() != true)
                return;

            var rootDirId = _coordinationModel.TypesByChildren.First(x => x.Value == _searchFolderType.Id).Key;
            CreateSearchSetsFromXml(rootDirId, dialog.FileName);

            var errors = _parseErrors.ToString();
            if (string.IsNullOrWhiteSpace(errors)) 
                return;

            var errorFilePath = Path.GetTempFileName();
            File.WriteAllText(errorFilePath, errors);
            MessageBox.Show($"Import finished with errors, see log: {errorFilePath}. Close dialog for copying log file path to clipboard");
            Clipboard.SetText(errorFilePath);
            _parseErrors.Clear();
        }

        private void CreateSearchSetsFromXml(Guid rootDirId, string fileName)
        {
            var reader = new StreamReader(fileName);
            if (!(ExchangeSerializer.Deserialize(reader) is Exchange exchange))
                return;

            var directoryId = Guid.NewGuid();
            _objectModifier
                .CreateById(directoryId, rootDirId, _searchFolderType)
                .SetAttribute(AttributeNames.Name, $"{Path.GetFileNameWithoutExtension(fileName)}");

            if (exchange.Selectionsets != null)
            {
                var selectionSets = exchange.Selectionsets;
                if (selectionSets.Selectionset != null)
                {
                    foreach (var item in selectionSets.Selectionset)
                    {
                        var searchSetBuilder = ParseNSet(item);

                        var searchSetString = new SearchSetSerializer().SerializeToJson(searchSetBuilder.Build());

                        _objectModifier.CreateById(Guid.NewGuid(), directoryId, _searchSetType)
                            .SetAttribute(AttributeNames.Name, item.Name)
                            .SetAttribute(AttributeNames.SearchRequest, searchSetString);
                    }
                }

                if (selectionSets.Viewfolder != null)
                {
                    foreach (var item in selectionSets.Viewfolder)
                        ParseNFolder(item, directoryId);
                }
            }

            if (exchange.Batchtest != null)
            {
                var selectionSets = exchange.Batchtest.Selectionsets;
                if (selectionSets != null)
                {
                    if (selectionSets.Selectionset != null)
                    {
                        foreach (var item in selectionSets.Selectionset)
                        {
                            var searchSetBuilder = ParseNSet(item);

                            var searchSetString = new SearchSetSerializer().SerializeToJson(searchSetBuilder.Build());

                            _objectModifier.CreateById(Guid.NewGuid(), directoryId, _searchSetType)
                                .SetAttribute(AttributeNames.Name, item.Name)
                                .SetAttribute(AttributeNames.SearchRequest, searchSetString);
                        }
                    }

                    if (selectionSets.Viewfolder != null)
                    {
                        foreach (var item in selectionSets.Viewfolder)
                            ParseNFolder(item, directoryId);
                    }
                }
            }

            reader.Close();
            _objectModifier.Apply();
        }

        private void ParseNFolder(Viewfolder folder, Guid parentId)
        {
            var newFolderId = Guid.NewGuid();
            _objectModifier.CreateById(newFolderId, parentId, _searchFolderType)
                .SetAttribute(AttributeNames.Name, folder.Name);

            if (folder.Selectionset != null)
            {
                foreach (var item in folder.Selectionset)
                {
                    var searchSetBuilder = ParseNSet(item);

                    var searchSetString = new SearchSetSerializer().SerializeToJson(searchSetBuilder.Build());

                    _objectModifier.CreateById(Guid.NewGuid(), newFolderId, _searchSetType)
                        .SetAttribute(AttributeNames.Name, item.Name)
                        .SetAttribute(AttributeNames.SearchRequest, searchSetString);
                }
            }

            if (folder.ViewfolderProperty != null)
            {
                foreach (var item in folder.ViewfolderProperty)
                    ParseNFolder(item, newFolderId);
            }
        }

        private IModelSearchSetBuilder ParseNSet(Selectionset set)
        {
            var andExpressions = new List<IModelSearchExpression>();
            var orExpressions = new List<IModelSearchExpression>();
            IModelSearchSetBuilder searchSetBuilder = null;
            (IModelSearchExpression Expression, SearchSetLogicalOperator SearchSetLogicalOperator) last = (null, SearchSetLogicalOperator.And);

            for (var i = 0; i < set.Findspec.Conditions.Count; i++)
            {
                var condition = set.Findspec.Conditions[i];

                var propCategory = condition.Category?.Name.Value;
                var searchProperty = ModelProperty.WithName(condition.Property.Name.Value, propCategory, PropertyDataType.String);
                var expression = GetExpression(searchProperty, condition.Test, condition.Flags, condition.Value.Data);

                if (i == 0)
                {
                    last.Expression = expression;
                    continue;
                }

                var flags = (NavisFlags)condition.Flags;
                if (flags.HasFlag(NavisFlags.Or))
                {
                    if (andExpressions.Any())
                    {
                        if (searchSetBuilder == null)
                            searchSetBuilder = ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.Or);

                        FillSearchSetBuilder(andExpressions, new List<IModelSearchExpression>(), searchSetBuilder);

                        andExpressions.Clear();
                        last = (expression, SearchSetLogicalOperator.Or);
                        continue;
                    }

                    if (last.Expression != null)
                        orExpressions.Add(last.Expression);

                    last = (expression, SearchSetLogicalOperator.Or);
                    continue;
                }

                if (orExpressions.Any())
                {
                    if (searchSetBuilder == null)
                        searchSetBuilder = ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.Or);

                    FillSearchSetBuilder(new List<IModelSearchExpression>(), orExpressions, searchSetBuilder);
                    orExpressions.Clear();
                }

                if (last.Expression != null)
                    andExpressions.Add(last.Expression);

                if (expression != null)
                    andExpressions.Add(expression);

                last = (null, SearchSetLogicalOperator.And);
            }

            if (last.Expression != null)
            {
                if (last.SearchSetLogicalOperator == SearchSetLogicalOperator.Or)
                    orExpressions.Add(last.Expression);
                else
                    andExpressions.Add(last.Expression);
            }

            if (searchSetBuilder != null)
                return FillSearchSetBuilder(andExpressions, orExpressions, searchSetBuilder);
            
            if (!andExpressions.Any() & orExpressions.Any() || andExpressions.Any() & !orExpressions.Any())
                return FillSearchSetBuilder(andExpressions, orExpressions);
            
            return FillSearchSetBuilder(andExpressions, orExpressions, ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.And));
        }

        private static IModelSearchSetBuilder FillSearchSetBuilder(List<IModelSearchExpression> andExpressions, List<IModelSearchExpression> orExpressions, IModelSearchSetBuilder setBuilder = null)
        {
            var expressionsBuilder = ModelSearchExpressionsBuilder.CreateBuilder();

            if (setBuilder == null)
            {
                if (andExpressions.Any())
                {
                    andExpressions.ForEach(andEx => expressionsBuilder.AddExpression(andEx));
                    return ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.And).AddExpressions(expressionsBuilder);
                }

                orExpressions.ForEach(orEx => expressionsBuilder.AddExpression(orEx));
                return ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.Or).AddExpressions(expressionsBuilder);
            }

            if (andExpressions.Any())
            {
                var andExpressionBuilder = ModelSearchExpressionsBuilder.CreateBuilder();
                andExpressions.ForEach(andEx => andExpressionBuilder.AddExpression(andEx));

                if (andExpressions.Count > 1 && setBuilder.Build().RootGroup.GroupingOperator != SearchSetLogicalOperator.And)
                    setBuilder.NewGroup(SearchSetLogicalOperator.And).AddExpressions(andExpressionBuilder);
                else
                    setBuilder.AddExpressions(andExpressionBuilder);
            }

            if (orExpressions.Any())
            {
                var orExpressionBuilder = ModelSearchExpressionsBuilder.CreateBuilder();
                orExpressions.ForEach(orEx => orExpressionBuilder.AddExpression(orEx));

                if (orExpressions.Count > 1 && setBuilder.Build().RootGroup.GroupingOperator != SearchSetLogicalOperator.Or)
                    setBuilder.NewGroup(SearchSetLogicalOperator.Or).AddExpressions(orExpressionBuilder);
                else
                    setBuilder.AddExpressions(orExpressionBuilder);
            }

            return setBuilder;
        }

        private IModelSearchExpression GetExpression(ModelSearchProperty searchProperty, FindspectestType specType, int flag, Data data)
        {
            var value = data.Text?.First();
            
            if (value == null)
                return null;

            var flags = (NavisFlags)flag;
            if (flags.HasFlag(NavisFlags.Not))
            {
                switch (specType)
                {
                    case FindspectestType.Equals:
                        return searchProperty.NotEqualTo(value);

                    case FindspectestType.NotEquals:
                        return searchProperty.EqualTo(value);

                    case FindspectestType.Contains:
                        return searchProperty.NotContains(value);

                    case FindspectestType.Prop:
                        return searchProperty.NotDefined();

                    case FindspectestType.NoProp:
                        return searchProperty.Defined();

                    case FindspectestType.GreaterEqual:
                        return searchProperty.LessThan(double.Parse(value, CultureInfo.InvariantCulture));

                    case FindspectestType.GreaterThan:
                        return searchProperty.LessThanOrEqual(double.Parse(value, CultureInfo.InvariantCulture));

                    case FindspectestType.LessEqual:
                        return searchProperty.GreaterThan(double.Parse(value, CultureInfo.InvariantCulture));

                    case FindspectestType.LessThan:
                        return searchProperty.GreaterThanOrEqual(double.Parse(value, CultureInfo.InvariantCulture));

                    case FindspectestType.Never:
                    case FindspectestType.Attrib:
                    case FindspectestType.NoAttrib:
                    case FindspectestType.SameType:
                    case FindspectestType.Wildcard:
                    case FindspectestType.WithinDay:
                    case FindspectestType.WithinWeek:
                    default:
                        _parseErrors.AppendLine($"not supported type: '{specType}' with negative flag. Category: '{searchProperty.CategoryName}', name: '{searchProperty.Name}'. Value: '{value}'");
                        return null;
                }
            }

            switch (specType)
            {
                case FindspectestType.Equals:
                    return searchProperty.EqualTo(value);

                case FindspectestType.NotEquals:
                    return searchProperty.NotEqualTo(value);

                case FindspectestType.Contains:
                    return searchProperty.Contains(value);

                case FindspectestType.Wildcard:
                    return searchProperty.Wildcard(value);

                case FindspectestType.GreaterEqual:
                    return searchProperty.GreaterThanOrEqual(double.Parse(value, CultureInfo.InvariantCulture));

                case FindspectestType.LessEqual:
                    return searchProperty.LessThanOrEqual(double.Parse(value, CultureInfo.InvariantCulture));

                case FindspectestType.GreaterThan:
                    return searchProperty.GreaterThan(double.Parse(value, CultureInfo.InvariantCulture));

                case FindspectestType.LessThan:
                    return searchProperty.LessThan(double.Parse(value, CultureInfo.InvariantCulture));

                case FindspectestType.Prop:
                    return searchProperty.Defined();

                case FindspectestType.NoProp:
                    return searchProperty.NotDefined();

                case FindspectestType.Never:
                case FindspectestType.Attrib:
                case FindspectestType.NoAttrib:
                case FindspectestType.SameType:
                case FindspectestType.WithinDay:
                case FindspectestType.WithinWeek:
                default:
                    _parseErrors.AppendLine($"not supported typ: '{specType}'. Category: '{searchProperty.CategoryName}', name: '{searchProperty.Name}'. Value: '{value}'");
                    return null;
            }
        }

        private void SubscribeForModelUpdates(Guid id, Action<IDataObject> action)
        {
            _subscription?.Dispose();

            var observable = _objectsRepository.SubscribeObjects(new[] { id });
            _subscription = observable.Where(x => x.State == DataState.Loaded
                                                      && x.SynchronizationState == SynchronizationState.Synchronized).Subscribe(action);
        }
        private void OnModelUpdated(IDataObject modelObj)
        {
            _coordinationModel = modelObj;
        }

        public void Dispose()
        {
            _subscription?.Dispose();
        }
    }
}
