﻿using System;
using System.Linq;
using System.Windows;
using System.Collections.Generic;
using Ascon.Pilot.Bim.SDK.ModelTab;
using Ascon.Pilot.Bim.SDK.ModelViewer;
using Ascon.Pilot.Bim.SDK.ModelTab.Toolbar;

namespace Ascon.Pilot.Bim.SDK.ToolbarSample
{
    public class ToolbarItemsController : IDisposable
    {
        private IModelTab _modelTab;
        private readonly List<IToolbarCommandItem> _toolbarCommands = new List<IToolbarCommandItem>();
        private readonly byte[] _icon;

        public ToolbarItemsController(IModelTab modelTab)
        {
            _modelTab = modelTab;
            _icon = IconLoader.GetIcon("toolbar_icon.svg");
        }

        public void AddToolbarItems()
        {
            var toolbarCommands = new List<IToolbarCommandItem>();
            var toolBarManager = _modelTab.GetToolbarManager();

            var buttonItem = AddToolbarButtonItem(toolBarManager);
            toolbarCommands.Add(buttonItem);

            var toggleItem = AddToolbarToggleItem(toolBarManager);
            toolbarCommands.Add(toggleItem);

            var menuItem = AddToolbarMenuItem(toolBarManager);
            toolbarCommands.Add(menuItem);

            _toolbarCommands.AddRange(toolbarCommands);
        }

        public IModelViewer ModelViewer { get; private set; }

        private IToolbarButtonItem AddToolbarButtonItem(IToolbarManager toolBarManager)
        {
            var buttonItem = toolBarManager.AddToolbarButtonItem(toolBarManager.GetItems().Count());
            buttonItem.Title = "Button item";
            buttonItem.IsEnabled = false;
            buttonItem.Icon = _icon;
            buttonItem.IsTitleVisible = false;
            buttonItem.Tooltip = "Button item";
            buttonItem.OnClick = () => MessageBox.Show("Button item clicked");

            return buttonItem;
        }

        private IToolbarToggleItem AddToolbarToggleItem(IToolbarManager toolBarManager)
        {
            var toggleButton = toolBarManager.AddToolbarToggleItem(toolBarManager.GetItems().Count());
            toggleButton.Title = "Toggle item";
            toggleButton.IsEnabled = false;
            toggleButton.Tooltip = "Toggle item";
            toggleButton.OnClick = isChecked => MessageBox.Show(isChecked ? "toggle button is checked" : "toggle button is unchecked");

            return toggleButton;
        }

        private IToolbarMenuItem AddToolbarMenuItem(IToolbarManager toolBarManager)
        {
            var menuItem = toolBarManager.AddToolbarMenuItem(toolBarManager.GetItems().Count());
            menuItem.Title = "Menu item";
            menuItem.IsEnabled = false;
            menuItem.Icon = _icon;
            menuItem.IsTitleVisible = false;

            var buttonFromMenu = menuItem.AddToolbarButtonItem(0);
            buttonFromMenu.Title = "Button item from menu";
            buttonFromMenu.IsEnabled = true;
            buttonFromMenu.OnClick = () => MessageBox.Show("Button from menu is clicked");

            menuItem.AddSeparator(1);

            var toggleButtonFromMenu = menuItem.AddToolbarToggleItem(2);
            toggleButtonFromMenu.Title = "Toggle item from menu";
            toggleButtonFromMenu.IsEnabled = true;
            toggleButtonFromMenu.OnClick = isChecked =>
                MessageBox.Show(isChecked ? "toggle button from menu is checked" : "toggle button from menu is unchecked");

            return menuItem;
        }

        public void OnModelLoaded(IModelViewer modelViewer)
        {
            if (_modelTab?.Id != modelViewer.TabId)
                return;

            ModelViewer = modelViewer;
            _toolbarCommands.ForEach(x => x.IsEnabled = true);
        }

        public void OnModelClosed(IModelViewer modelViewer)
        {
            if (_modelTab?.Id != modelViewer.TabId)
                return;

            ModelViewer = null;
            _toolbarCommands.ForEach(x => x.IsEnabled = false);
        }

        public void Dispose()
        {
            _toolbarCommands.Clear();
            _modelTab = null;
        }
    }
}
