﻿using Ascon.Pilot.SDK;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;

namespace BcfApiService.Loaders
{
    public class ObjectLoader : IObserver<IDataObject>
    {
        private readonly IObjectsRepository _repository;
        private readonly Dictionary<Guid, (IDisposable subscription, TaskCompletionSource<IDataObject> tcs)> _completions;
        private long _changesetId;

        public ObjectLoader(IObjectsRepository repository)
        {
            _repository = repository;
            _completions = new  Dictionary<Guid, (IDisposable subscription, TaskCompletionSource<IDataObject> tcs)>();
        }

        public Task<IDataObject> Load(Guid id, long changesetId = 0)
        {
            _changesetId = changesetId;
            var subscription = _repository.SubscribeObjects(new[] { id }).Subscribe(this);
            var tcs = new TaskCompletionSource<IDataObject>(TaskCreationOptions.RunContinuationsAsynchronously);
            _completions[id] = (subscription, tcs);
            return tcs.Task;
        }

        public void OnNext(IDataObject value)
        {
            if (value.State != DataState.Loaded)
                return;

            if (value.LastChange() < _changesetId)
                return;

            if (_completions.TryGetValue(value.Id, out var completion)) 
            {
                completion.tcs.TrySetResult(value);
                completion.subscription.Dispose();
                _completions.Remove(value.Id);
            }
        }

        public void OnError(Exception error) { }

        public void OnCompleted() { }
    }
}
