﻿using Ascon.Pilot.SDK;
using BcfApiService.DTOs;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using System;
using Ascon.Pilot.SDK.Data;

namespace BcfApiService.Services
{
    public class CommentService : Service
    {
        private readonly IMessagesRepository _messagesRepository;

        public CommentService(IObjectsRepository objectsRepository, ISearchService searchService, IMessagesRepository messagesRepository) : base(objectsRepository, searchService) 
        {
            _messagesRepository = messagesRepository;
        }

        public async Task<IEnumerable<CommentDTO>> GetComments(Guid projectId, Guid topicId)
        {
            var resultList = new List<CommentDTO>();
            ISearchResult searchResult = await _objectIdsLoader.Load(_modelPartType.Id);

            if (!searchResult.Result.Contains(projectId))
            {
                return resultList;
            }
            else
            {
                IChat chat = await _messagesRepository.LoadChatAsync(topicId);

                if (chat.Id != topicId)
                {
                    return resultList;
                }

                IEnumerable<IChatMessage> messages = await _messagesRepository.LoadMessagesAsync(chat.Id, DateTime.MinValue, DateTime.MaxValue, int.MaxValue);
                messages = messages
                    .Where(message => message.Type == MessageType.TextMessage || message.Type == MessageType.MessageAnswer)
                    .OrderBy(message => message.ClientDateUtc);

                foreach (IChatMessage message in messages)
                {
                    IPerson messageSender = _personLoader.LoadPerson(message.CreatorId);
                    if (message.RelatedMessages.Any(relatedMessage => relatedMessage.Type == MessageType.EditTextMessage))
                    {
                        IChatMessage lastVersionMessage = message.RelatedMessages
                            .Where(relatedMessage => relatedMessage.Type == MessageType.EditTextMessage)
                            .OrderByDescending(relatedMessage => relatedMessage.ClientDateUtc)
                            .First();
                        
                        resultList.Add(new CommentDTO
                        {
                            Guid = lastVersionMessage.Id.ToString().ToUpperInvariant(),
                            Date = message.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                            Author = messageSender.Login,
                            Comment = lastVersionMessage.GetData<ITextMessageData>().Text,
                            TopicGuid = topicId.ToString().ToUpperInvariant(),
                            ModifiedDate = lastVersionMessage.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                            ModifiedAuthor = messageSender.Login,
                        });
                    }
                    else
                    {
                        resultList.Add(new CommentDTO
                        {
                            Guid = message.Id.ToString().ToUpperInvariant(),
                            Date = message.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                            Author = messageSender.Login,
                            Comment = message.GetData<ITextMessageData>().Text,
                            TopicGuid = topicId.ToString().ToUpperInvariant(),
                        });
                    }
                }

                return resultList;
            }
        }

        public async Task<CommentDTO> FindCommentById(Guid projectId, Guid topicId, Guid commentId)
        {
            ISearchResult searchResult = await _objectIdsLoader.Load(_modelPartType.Id);

            if (!searchResult.Result.Contains(projectId))
            {
                return null;
            }
            else
            {
                IChat chat = await _messagesRepository.LoadChatAsync(topicId);

                if (chat.Id != topicId)
                {
                    return null;
                }

                IEnumerable<IChatMessage> messages = await _messagesRepository.LoadMessagesAsync(chat.Id, DateTime.MinValue, DateTime.MaxValue, int.MaxValue);
                messages = messages.Where(message => message.Type == MessageType.TextMessage || message.Type == MessageType.MessageAnswer);
                
                IChatMessage messageToFind = messages.FirstOrDefault(message => message.Id == commentId);
                if (messageToFind != null)
                {
                    IPerson messageSender = _personLoader.LoadPerson(messageToFind.CreatorId);
                    return new CommentDTO
                    {
                        Guid = messageToFind.Id.ToString().ToUpperInvariant(),
                        Date = messageToFind.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                        Author = messageSender.Login,
                        Comment = messageToFind.GetData<ITextMessageData>().Text,
                        TopicGuid = topicId.ToString().ToUpperInvariant(),
                    };
                }
                else
                {
                    IChatMessage lastVersionMessageToFind = messages
                        .Where(message => message.RelatedMessages.Any(relatedMessage => relatedMessage.Type == MessageType.EditTextMessage))
                        .Select(message => message.RelatedMessages
                            .Where(relatedMessage => relatedMessage.Type == MessageType.EditTextMessage)
                            .OrderByDescending(relatedMessage => relatedMessage.ClientDateUtc)
                            .First())
                        .FirstOrDefault(message => message.Id == commentId);

                    if (lastVersionMessageToFind != null)
                    {
                        IChatMessage firstVersionMessage = messages.First(message => message.Id == lastVersionMessageToFind.RelatedMessageId);
                        IPerson messageSender = _personLoader.LoadPerson(firstVersionMessage.CreatorId);
                        return new CommentDTO
                        {
                            Guid = lastVersionMessageToFind.Id.ToString().ToUpperInvariant(),
                            Date = firstVersionMessage.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                            Author = messageSender.Login,
                            Comment = lastVersionMessageToFind.GetData<ITextMessageData>().Text,
                            TopicGuid = topicId.ToString().ToUpperInvariant(),
                            ModifiedDate = lastVersionMessageToFind.ClientDateUtc.ToLocalTime().ToString("yyyy-MM-ddTHH:mm:ss.fffZ"),
                            ModifiedAuthor = messageSender.Login,
                        };
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }
    }
}
