﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;

namespace BcfApiService
{
    public class WebServerMethod
    {
        public static int GET = 0;
        public static int POST = 1;
    }

    public class WebServerResponse
    {

        private HttpListenerResponse _response;

        public WebServerResponse(HttpListenerContext context)
        {
            _response = context.Response;
        }

        public void WriteResponse(string message)
        {
            try
            {
                byte[] buf = Encoding.UTF8.GetBytes(message);
                _response.ContentLength64 = buf.Length;
                _response.OutputStream.Write(buf, 0, buf.Length);
            }
            catch (Exception e)
            {
                LogManager.GetLogger(typeof(WebServerResponse)).Error(e);
            }
            finally
            {
                _response.OutputStream.Close();
            }
        }

        public void SetStatusCodeAndDescription(int statusCode, string description)
        {
            _response.StatusCode = statusCode;
            _response.StatusDescription = description;
            _response.Close();
        }

        public void SetMIMEtype(string MIMEtype)
        {
            _response.ContentType = MIMEtype;
        }
    }

    public class WebServerRequest
    {
        private HttpListenerRequest _request;

        public WebServerRequest(HttpListenerContext context)
        {
            _request = context.Request;
        }

        public string GetQueryStringParameter(string key)
        {
            if (_request.QueryString.AllKeys.Contains(key))
                return _request.QueryString[key];
            return null;
        }

        public string GetUrlAbsolutePath()
        {
            return _request.Url.AbsolutePath;
        }
    }

    public class WebServer
    {
        private HttpListener _httpSocket = new HttpListener();
        private Dictionary<Regex, Action<WebServerRequest, WebServerResponse>>[] _actionRoutes = new Dictionary<Regex, Action<WebServerRequest, WebServerResponse>>[2];
        private readonly string _prefix;

        public WebServer() : this(8080) { }

        public WebServer(int port)
        {
            _prefix = string.Format("http://localhost:{0}/", port);
            _httpSocket.Prefixes.Add(_prefix);
        }


        public void GetMethod(Regex path, Action<WebServerRequest, WebServerResponse> method)
        {
            AddRoute(WebServerMethod.GET, path, method);
        }

        public void PostMethod(Regex url, Action<WebServerRequest, WebServerResponse> method)
        {
            AddRoute(WebServerMethod.POST, url, method);
        }

        public void Start()
        {
            _httpSocket.Start();

            ThreadPool.QueueUserWorkItem((o) =>
            {
                try
                {
                    while (_httpSocket.IsListening)
                    {
                        ThreadPool.QueueUserWorkItem((context) =>
                        {
                            var httpContext = context as HttpListenerContext;

                            Action<WebServerRequest, WebServerResponse> activeMethod = null;

                            switch (httpContext.Request.HttpMethod)
                            {
                                case "GET":
                                    activeMethod = FindRoute(WebServerMethod.GET, httpContext.Request.Url.AbsolutePath);
                                    break;
                                case "POST":
                                    activeMethod = FindRoute(WebServerMethod.POST, httpContext.Request.Url.AbsolutePath);
                                    break;
                                default:
                                    activeMethod = MethodNotImplemented;
                                    throw new Exception();
                            }

                            activeMethod(new WebServerRequest(httpContext), new WebServerResponse(httpContext));

                        }, _httpSocket.GetContext());
                    }
                }
                catch (Exception e)
                {
                    LogManager.GetLogger(typeof(WebServerResponse)).Error(e);
                }
            });
        }

        public void Stop()
        {
            _httpSocket.Stop();
            _httpSocket.Close();
        }

        private void AddRoute(int httpMethod, Regex route, Action<WebServerRequest, WebServerResponse> method)
        {
            if (_actionRoutes[httpMethod] == null)
            {
                _actionRoutes[httpMethod] = new Dictionary<Regex, Action<WebServerRequest, WebServerResponse>>();
            }

            _actionRoutes[httpMethod].Add(route, method);
        }

        private Action<WebServerRequest, WebServerResponse> FindRoute(int httpMethod, string uri)
        {
            Dictionary<Regex, Action<WebServerRequest, WebServerResponse>> _routes = _actionRoutes[httpMethod];

            foreach (Regex route in _routes.Keys)
            {
                if (route.IsMatch(uri))
                    return _routes[route];
            }

            return NotFoundResponse;
        }

        private static void NotFoundResponse(WebServerRequest request, WebServerResponse response)
        {
            response.SetStatusCodeAndDescription(404, "Not Found");
        }

        private static void MethodNotImplemented(WebServerRequest request, WebServerResponse response)
        {
            response.SetStatusCodeAndDescription(401, "Not Implemented");
        }
    }
}
