/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.SDK.AccessRightsSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class Main : IMenu<ObjectsViewContext>
    {
        private readonly IPilotDialogService _dialogService;
        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;
        private IDataObject _selected;

        [ImportingConstructor]
        public Main(IPilotDialogService dialogService, IObjectModifier modifier, IObjectsRepository repository)
        {
            _dialogService = dialogService;
            _modifier = modifier;
            _repository = repository;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();

            var itemNames = builder.ItemNames.ToList();
            const string indexItemName = "miShowSharingSettings";
            var insertIndex = itemNames.IndexOf(indexItemName) + 1;

            // Show menu when a document is selected
            if (objects.Count != 1) 
                return;
            
            builder.AddItem("miImproveAccessRights", insertIndex).WithHeader("Improve access rights...");
            _selected = objects.First();

            if (_selected.IsSecret)
                builder.AddItem("miMakePublic", ++insertIndex).WithHeader("Make public");
            else
            {
                var myAccess = GetMyAccessLevel(_selected);
                if (myAccess == AccessLevel.ViewEdit)
                    builder.AddItem("miMakeSecret", ++insertIndex).WithHeader("Make secret");
            }
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name.Equals("miMakePublic"))
            {
                var builder = _modifier.EditById(_selected.Id);
                builder.MakePublic();
                _modifier.Apply();
                return;
            }

            if (name.Equals("miMakeSecret"))
            {
                var builder = _modifier.EditById(_selected.Id);
                builder.MakeSecret();
                _modifier.Apply();
                return;
            }

            var orgUnit = _dialogService.ShowPositionSelectorDialog().FirstOrDefault();
            if (orgUnit == null)
                return;

            if (name.Equals("miImproveAccessRights"))
            {
                var accessLevel = GetAccessLevel(_selected, orgUnit.Id);
                var improveAccessLevel = GetMyAccessLevel(_selected);
                if (accessLevel == improveAccessLevel)
                    return;

                var builder = _modifier.EditById(_selected.Id);
                builder.AddAccessRecords(orgUnit.Id, improveAccessLevel, DateTime.MaxValue, AccessInheritance.None, AccessType.Allow);
                _modifier.Apply();
            }
        }

        private AccessLevel GetMyAccessLevel(IDataObject element)
        {
            var currentAccesLevel = AccessLevel.None;
            var person = _repository.GetCurrentPerson();
            foreach (var position in person.Positions)
            {
                currentAccesLevel = currentAccesLevel | GetAccessLevel(element, position.Position);
            }

            return currentAccesLevel;
        }

        private AccessLevel GetAccessLevel(IDataObject element, int positonId)
        {
            var currentAccesLevel = AccessLevel.None;
            var orgUnits = _repository.GetOrganisationUnits().ToDictionary(k => k.Id);
            var accesses = GetAccessRecordsForPosition(element, positonId, orgUnits);
            foreach (var source in accesses)
            {
                currentAccesLevel = currentAccesLevel | source.Access.AccessLevel;
            }
            return currentAccesLevel;
        }

        private IEnumerable<IAccessRecord> GetAccessRecordsForPosition(IDataObject obj, int positionId, IDictionary<int, IOrganisationUnit> organisationUnits)
        {
            return obj.Access2.Where(x => BelongsTo(positionId, x.OrgUnitId, organisationUnits));
        }

        public static bool BelongsTo(int position, int organisationUnit, IDictionary<int, IOrganisationUnit> organisationUnits)
        {
            Stack<int> units = new Stack<int>();
            units.Push(organisationUnit);
            while (units.Any())
            {
                var unitId = units.Pop();
                if (position == unitId)
                    return true;

                IOrganisationUnit unit;
                if (organisationUnits.TryGetValue(unitId, out unit))
                {
                    foreach (var childUnitId in unit.Children)
                    {
                        units.Push(childUnitId);
                    }
                }
            }
            return false;
        }
    }
}
