/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.Threading;
using System.Diagnostics;

namespace Ascon.Pilot.SDK.ClearCacheDaemon
{
    [Export(typeof(IDataPlugin))]
    public class Daemon : IDataPlugin
    {
        private readonly IObjectsRepository _repository;
        private readonly IFileProvider _fileProvider;
        private readonly IPilotDialogService _dialogService;
        private readonly Timer _timer;

        [ImportingConstructor]
        public Daemon(IObjectsRepository repository, IFileProvider fileProvider, IPilotDialogService dialogService)
        {
            _repository = repository;
            _fileProvider = fileProvider;
            _dialogService = dialogService;

            _timer = new Timer(OnTick, null, dueTime: TimeSpan.FromSeconds(10), period: TimeSpan.FromHours(10));
        }

        private async void OnTick(object state)
        {
            var ids = await _repository.LoadCleanableFileIds();
            long cleanedBytes = 0;
            foreach (var id in ids)
            {
                try
                {
                    if(!_fileProvider.Exists(id))
                        continue;

                    var size = _fileProvider.GetFileSizeOnDisk(id);
                    _fileProvider.DeleteLocalFile(id);
                    cleanedBytes += size;
                }
                catch
                {
                    Debug.WriteLine($"Unable to delete file {id}");
                }
            }
            _dialogService.ShowBalloon("Local cache cleaning", $"Freed {cleanedBytes} bytes on disk", PilotBalloonIcon.Info);
        }
    }
}
