/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.Linq;
using System.Windows.Media;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.FileSnapshotSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class FileSnapshotSample : IMenu<ObjectsViewContext>
    {
        private readonly IObjectsRepository _repository;
        private readonly IObjectModifier _modifier;
        private Guid _selectedId;
        private readonly SignatureRequestBuilder _signatureRequestBuilder;

        [ImportingConstructor]
        public FileSnapshotSample(IPilotDialogService dialogService, IObjectsRepository repository, IObjectModifier modifier, IFileProvider fileProvider)
        {
            _repository = repository;
            _modifier = modifier;
            var convertFromString = ColorConverter.ConvertFromString(dialogService.AccentColor);
            if (convertFromString != null)
            {
                var accentColor = (Color)convertFromString;
                ColorScheme.Initialize(accentColor, dialogService.Theme);
            }

            _signatureRequestBuilder = new SignatureRequestBuilder(dialogService, modifier, repository.GetDatabaseId(), fileProvider);
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();
            if (objects.Count != 1)
                return;

            var current = objects.Single();
            if (!current.Type.HasFiles) 
                return;
            
            builder.AddItem("MenuCommand", 0).WithHeader("Show document versions");
            builder.AddItem("MenuAddSignaturesCommand", 1).WithHeader("Add virtual signature requests");

            var files = current.ActualFileSnapshot.Files;
            if (files.Any())
                if (files.FirstOrDefault(f => f.SignatureRequests.Any()) != null)
                {
                    builder.AddItem("MenuRemoveSignaturesCommand", 1).WithHeader("Remove virtual signature requests");
                }

            _selectedId = current.Id;
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name == "MenuCommand")
            {
                var view = new DocumentHistoryView();
                var model = new DocumentHistoryViewModel(_selectedId, _repository, _modifier, view.Dispatcher);
                view.DataContext = model;
                view.ShowDialog();
            }

            if (name == "MenuAddSignaturesCommand")
                _signatureRequestBuilder.AddVirtualRequests(context.SelectedObjects.First());

            if (name == "MenuRemoveSignaturesCommand")
                _signatureRequestBuilder.RemoveVirtualRequests(context.SelectedObjects.First());

        }
    }
}
