/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace Ascon.Pilot.SDK.FileSnapshotSample
{
    class SignatureRequestBuilder
    {
        private readonly IPilotDialogService _dialogService;
        private readonly IObjectModifier _modifier;
        private readonly Guid _databaseId;
        private readonly IFileProvider _fileProvider;

        public SignatureRequestBuilder(IPilotDialogService dialogService, IObjectModifier modifier, Guid databaseId, IFileProvider fileProvider)
        {
            _dialogService = dialogService;
            _modifier = modifier;
            _databaseId = databaseId;
            _fileProvider = fileProvider;
        }

        public void AddVirtualRequests(IDataObject dataObject)
        {
            var currentFile = dataObject.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            if (currentFile == null)
                return;

            using (var stream = _fileProvider.OpenRead(currentFile))
            {
                var builder = _modifier.Edit(dataObject)
                    .CreateFileSnapshot("New virtual signature requests")
                    .AddFile(currentFile.Name, stream, currentFile.Created, currentFile.Accessed, currentFile.Modified);

                var signatureBuilder = builder.SetSignatures(p => FileExtensionHelper.IsXpsAlike(p.Name));

                var people = _dialogService.ShowPositionSelectorDialog();
                foreach (var unit in people)
                {
                    if (!unit.IsPosition)
                        continue;

                    signatureBuilder.Add(Guid.NewGuid())
                        .WithIsAdditional(true)
                        .WithPositionId(unit.Id)
                        .WithRequestSigner(unit.Title);
                }

                _modifier.Apply();
            }
        }

        public void RemoveVirtualRequests(IDataObject dataObject)
        {
            var currentFile = dataObject.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            if (currentFile == null)
                return;

            var signatureBuilder = _modifier.Edit(dataObject).SetSignatures(p => FileExtensionHelper.IsXpsAlike(p.Name));
            signatureBuilder.Remove(s => s.IsAdditional() && !((ISignatureRequest)s).Signs.Any());

            _modifier.Apply();
        }
    }

    internal class FileExtensionHelper
    {
        public const string XPS_ALIKE_REGEX_MASK = @"\.xps$|\.dwfx$";

        public static bool IsXpsAlike(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, XPS_ALIKE_REGEX_MASK);
        }

        private static bool IsFileMatchesExtensionMask(string filePath, string mask)
        {
            var extension = Path.GetExtension(filePath);
            return !string.IsNullOrEmpty(extension) && Regex.IsMatch(extension, mask, RegexOptions.IgnoreCase);
        }
    }
}
