/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel.Composition;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Remoting.Messaging;
using System.Threading.Tasks;
using System.Windows;
using Ascon.Pilot.SDK.Data;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.Controls;

namespace Ascon.Pilot.SDK.MessagesSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class MessagesSample : IMenu<ObjectsViewContext>
    {
        private readonly IMessagesRepository _repository;
        private readonly IMessageBuilder _messageBuilder;
        private readonly IObjectsRepository _objRepository;
        private readonly IPilotDialogService _dialogService;
        private const string LOAD_COMMAND = "LOAD_MESSAGES_COMMAND";

        [ImportingConstructor]
        public MessagesSample(IMessagesRepository repository, IMessageBuilder messageBuilder, IObjectsRepository objRepository, IPilotDialogService dialogService)
        {
            _repository = repository;
            _messageBuilder = messageBuilder;
            _objRepository = objRepository;
            _dialogService = dialogService;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var createItemBuilder = builder.AddItem(LOAD_COMMAND, 0).WithHeader("Messages");
        }

        public async void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name != LOAD_COMMAND)
                return;
            if(context.SelectedObjects.Count() != 1)
                return;

            var selectedObject = context.SelectedObjects.First();

            await LoadMessages(selectedObject);
        }

        private async Task LoadMessages(IDataObject selectedObject)
        {
            var chat = await _repository.LoadChatAsync(selectedObject.Id);

            var messages =
                await _repository.LoadMessagesAsync(selectedObject.Id, DateTime.MinValue, DateTime.MaxValue, Int32.MaxValue);

            var list = new List<MessageViewModel>();

            foreach (var chatMessage in messages.OrderBy(x => x.ServerDateUtc))
            {
                if (chatMessage.Type == MessageType.TextMessage)
                {
                    var edit = chatMessage.RelatedMessages.Where(x => x.Type == MessageType.EditTextMessage)
                        .OrderByDescending(x => x.ServerDateUtc).FirstOrDefault();

                    var text = "";
                    text = edit != null ? edit.GetData<ITextMessageData>().Text : chatMessage.GetData<ITextMessageData>().Text;

                    list.Add(new MessageViewModel()
                    {
                        Text = text,
                        Name = _objRepository.GetPerson(chatMessage.CreatorId)?.DisplayName
                    });
                }
            }

            var view = new MessagesView();
            var viewModel = new MessagesViewModel(_repository, selectedObject, chat.Id != Guid.Empty, _messageBuilder, _objRepository,_dialogService)
            {
                MessagesCollection = new ObservableCollection<MessageViewModel>(list)
            };

            view.DataContext = viewModel;

            var dialog = new PureWindow
            {
                Content = view,
                Title = "Сообщения по объекту",
                WindowStartupLocation = WindowStartupLocation.CenterScreen,
                Width = 800,
                Height = 600,
                ShowInTaskbar = true
            };
            dialog.Show();
        }
    }
}
