/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using Ascon.Pilot.Theme.Tools;

namespace Ascon.Pilot.SDK.ObjectsSample.ObjectCardView.ReferenceBook
{
    static class ReferenceBookComboBoxBehavior
    {
        public static readonly DependencyProperty ViewModelProperty =
            DependencyProperty.RegisterAttached("ViewModel", typeof(CardControlViewModel), typeof(ReferenceBookComboBoxBehavior), new PropertyMetadata(null, OnViewModelChanged));

        public static CardControlViewModel GetViewModel(DependencyObject obj)
        {
            return (CardControlViewModel)obj.GetValue(ViewModelProperty);
        }

        public static void SetViewModel(DependencyObject obj, CardControlViewModel value)
        {
            obj.SetValue(ViewModelProperty, value);
        }

        private static void OnViewModelChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var comboBox = d as ComboBox;
            if (comboBox == null)
                throw new NotSupportedException();

            var viewModel = (CardControlViewModel)e.NewValue;
            IReferenceBookConfiguration configuration;
            var success = viewModel.AttributeFormatParser.TryParseReferenceBookConfiguration(viewModel.Attribute.Configuration2(), out configuration);
           
            if(!success)
                return;

            comboBox.IsReadOnly = !configuration.IsEditable;

            comboBox.DropDownOpened += (o, a) =>
            {
                ComboBoxEx.SetIsLoading(comboBox, true);
                var loader = new ReferenceBookLoader(viewModel.Repository);
                loader.Completed += (l, args) =>
                {
                    var text = comboBox.Text;
                    comboBox.ItemsSource = GetCollectionView(configuration.Source, args.Objects, configuration.StringFormat, configuration.ElementsTypes.ToList(), viewModel.AttributeFormatParser, viewModel.Repository);
                    comboBox.Text = text;
                    ComboBoxEx.SetIsLoading(comboBox, false);
                };
                loader.Load(configuration.Source);
            };
            comboBox.SelectionChanged += (o, a) =>
            {
                var selectedItem = comboBox.SelectedItem as ReferenceBookItem;
                if (selectedItem != null)
                {
                    foreach (var attribute in selectedItem.Object.Attributes)
                    {
                        if (viewModel.Attribute.Name != attribute.Key)
                            viewModel.AutoComplete.Fill(attribute.Key, attribute.Value);
                    }
                }
            };
        }

        private static IEnumerable GetCollectionView(Guid sourceId, Dictionary<Guid, IDataObject> objects, string stringFormat, IList<string> typesToShow, IAttributeFormatParser attributeFormatParser, IObjectsRepository repository)
        {
            var items = new List<ReferenceBookItem>();
            BuildReferenceBookItems(sourceId, objects, new string[0], stringFormat, items, typesToShow, attributeFormatParser, repository);
            if (!items.Any())
                return null;

            var lcv = new ListCollectionView(items);
            for (int i = 0; i < items.Max(x => x.Categories.Count); i++)
            {
                lcv.GroupDescriptions.Add(new PropertyGroupDescription(string.Format("Categories[{0}]", i)));
            }
            return lcv;
        }

        private static void BuildReferenceBookItems(Guid parentId, Dictionary<Guid, IDataObject> objects, IList<string> categories, string stringFormat, List<ReferenceBookItem> result, IList<string> typesToShow, IAttributeFormatParser attributeFormatParser, IObjectsRepository repository)
        {
            var children = GetSortedChildren(parentId, objects, typesToShow, repository);
            if (children == null)
                return;

            foreach (var child in children)
            {
                var title = child.DisplayName;
                if (IsLeaf(child))
                {
                    var formattedTitle = string.IsNullOrEmpty(stringFormat)
                        ? title
                        : attributeFormatParser.AttributesFormat(stringFormat, child.Attributes.ToDictionary(x=>x.Key, x=>x.Value));
                    result.Add(new ReferenceBookItem(child, formattedTitle, categories.ToList()));
                }
                else
                {
                    BuildReferenceBookItems(child.Id, objects, categories.Union(new[] { title }).ToList(), stringFormat, result, typesToShow, attributeFormatParser, repository);
                }
            }
        }

        private static IEnumerable<IDataObject> GetSortedChildren(Guid parentId, Dictionary<Guid, IDataObject> objects, IList<string> typesToShow, IObjectsRepository repository)
        {
            IDataObject parent;
            if (!objects.TryGetValue(parentId, out parent))
                return null;

            var children = ChildrenFilters.GetChildrenForListView(parent, repository)
                .Where(objects.ContainsKey)
                .Select(x => objects[x]).Where(x => !typesToShow.Any() || !IsLeaf(x) || typesToShow.Contains(x.Type.Name))
                .ToList();
            children.Sort(new ObjectComparer());
            return children;
        }

        private static bool IsLeaf(IDataObject obj)
        {
            return !obj.Type.Children.Any();
        }
    }

    class ReferenceBookItem
    {
        public IDataObject Object { get; private set; }
        public string Title { get; private set; }
        public List<string> Categories { get; private set; }

        public ReferenceBookItem(IDataObject obj, string title, IEnumerable<string> categories)
        {
            if (obj == null)
                throw new ArgumentNullException("obj");
            if (title == null)
                throw new ArgumentNullException("title");
            if (categories == null)
                throw new ArgumentNullException("categories");
            Object = obj;
            Title = title;
            Categories = new List<string>(categories);
        }

        public override string ToString()
        {
            return Title;
        }
    }

    class ObjectComparer : IComparer<IDataObject>, IComparer
    {
        private readonly int _direction;

        public ObjectComparer(bool ascending = true)
        {
            _direction = ascending ? 1 : -1;
        }

        public int Compare(IDataObject x, IDataObject y)
        {
            var result = x.Type.Sort.CompareTo(y.Type.Sort);
            if (result != 0)
                return result;

            result = NaturalComparer.Compare(x.DisplayName, y.DisplayName);
            if (result != 0)
                return _direction * result;

            return _direction * (x.GetHashCode() - y.GetHashCode());
        }

        public int Compare(object x, object y)
        {
            var obj1 = x as IDataObject;
            var obj2 = y as IDataObject;
            return Compare(obj1, obj2);
        }
    }

    public class NaturalComparer
    {
        private static int ExtractNumber(string str, int index, out Int64 result)
        {
            result = 0;
            int i = index;
            while (i < str.Length)
            {
                if (Char.IsDigit(str[i]))
                {
                    result = result * 10 + (str[i] - '0');
                    i++;
                }
                else
                    break;
            }
            return i;
        }

        public static int Compare(string str1, string str2)
        {
            if (str1 == null && str2 == null)
                return 0;
            if (str1 == null)
                return -1;
            if (str2 == null)
                return 1;

            int i1 = 0;
            int i2 = 0;
            while (i1 < str1.Length && i2 < str2.Length)
            {
                if (Char.IsDigit(str1[i1]) && Char.IsDigit(str2[i2]))
                {
                    Int64 num1;
                    Int64 num2;
                    i1 = ExtractNumber(str1, i1, out num1);
                    i2 = ExtractNumber(str2, i2, out num2);
                    var result = num1.CompareTo(num2);
                    if (result != 0)
                        return result;
                    result = i2.CompareTo(i1);
                    if (result != 0)
                        return result;
                }
                else
                {
                    var leftChar = Char.ToUpper(str1[i1]);
                    var rightChar = Char.ToUpper(str2[i2]);
                    var result = leftChar.CompareTo(rightChar);
                    if (result != 0)
                        return result;
                    i1++;
                    i2++;
                }
            }
            if (i1 < str1.Length) return 1;
            if (i2 < str2.Length) return -1;
            return 0;
        }
    }
}
