/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.SDK.ContextMenuSample;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.Toolbar;
using Ascon.Pilot.Theme.Tools;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Windows;
using System.Windows.Automation;
using System.Windows.Controls;
using System.Windows.Threading;

namespace Ascon.Pilot.SDK.OpenDocumentInNewTabSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    [Export(typeof(IMenu<LinkedObjectsContext>))]
    [Export(typeof(IMenu<AttachmentManagerContext>))]
    [Export(typeof(IToolbar<ObjectsViewContext>))]
    [ExportMetadata("Name", "Open document in new tab command extension")]
    public class OpenDocumentInNewTabExtension : IMenu<ObjectsViewContext>, IMenu<LinkedObjectsContext>, IMenu<AttachmentManagerContext>, IToolbar<ObjectsViewContext>
    {
        private readonly ITabServiceProvider _tabServiceProvider;
        private const string MENU_ITEM_NAME = "openDocumentInNewTab";
        private const string MENU_ITEM_ICON = "new_tab_icon.svg";
        private const string MENU_ITEM_TITLE = "Открыть в новой вкладке";
        private const string CONFIRMATION_TITLE = "Подтверждение действия";
        private const string CONFIRMATION_STRING_FORMAT = "Вы действительно хотите открыть столько новых вкладок ({0} шт.)?";
        private Queue<Guid> _targetDocumentsIds;
        private Guid _currentTargetDocumentId;

        [ImportingConstructor]
        public OpenDocumentInNewTabExtension(ITabServiceProvider tabServiceProvider)
        {
            _tabServiceProvider = tabServiceProvider;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var commandIndex = builder.ItemNames.ToList().IndexOf("miSend") - 1;
            BuildMenuCore(builder, commandIndex, context.SelectedObjects.ToArray());
        }
        public void Build(IMenuBuilder builder, LinkedObjectsContext context)
        {
            var commandIndex = builder.ItemNames.ToList().IndexOf("miOpenAttachment");
            BuildMenuCore(builder, commandIndex, context.SelectedObjects.ToArray());
        }

        public void Build(IMenuBuilder builder, AttachmentManagerContext context)
        {
            var commandIndex = builder.ItemNames.ToList().IndexOf("miOpenAttachment");
            BuildMenuCore(builder, commandIndex, context.Attachments.ToArray());
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            OnMenuItemClickCore(name, context.SelectedObjects.ToArray());
        }

        public void OnMenuItemClick(string name, LinkedObjectsContext context)
        {
            OnMenuItemClickCore(name, context.SelectedObjects.ToArray());
        }

        public void OnMenuItemClick(string name, AttachmentManagerContext context)
        {
            OnMenuItemClickCore(name, context.Attachments.ToArray());
        }

        public void Build(IToolbarBuilder builder, ObjectsViewContext context)
        {
            var array = context.SelectedObjects.ToArray();
            var dataObject = context.IsContext || array.Length != 1 ? null : array[0];
            if (dataObject == null)
                return;

            OnDocumentLoaded(dataObject.Id);
        }

        public void OnToolbarItemClick(string name, ObjectsViewContext context)
        {
        }

        private void BuildMenuCore(IMenuBuilder builder, int commandIndex, IDataObject[] targets)
        {
            _targetDocumentsIds?.Clear();

            if (targets == null || !targets.All(IsValidTarget))
                return;

            var menuBuilder = builder;

            var index = commandIndex < 0 ? 0 : commandIndex + 1;

            menuBuilder.AddItem(MENU_ITEM_NAME, index)
                .WithHeader(MENU_ITEM_TITLE)
                .WithIcon(IconLoader.GetIcon(MENU_ITEM_ICON));
        }

        private bool IsValidTarget(IDataObject target)
        {
            return target.Type.HasFiles;
        }

        private void OnMenuItemClickCore(string name, IDataObject[] targets)
        {
            if (name != MENU_ITEM_NAME
                || targets.Length > 3 
                && MessageBox.Show(
                    string.Format(CONFIRMATION_STRING_FORMAT, targets.Length),
                    CONFIRMATION_TITLE, 
                    MessageBoxButton.YesNo) != MessageBoxResult.Yes)
                return;

            _targetDocumentsIds = new Queue<Guid>(targets.Select(x => x.Id));
            OpenTargetInNewTab();
        }

        private void OpenTargetInNewTab()
        {
            if (!_targetDocumentsIds.Any())
                return;

            _currentTargetDocumentId = _targetDocumentsIds.Dequeue();
            _tabServiceProvider.ShowElement(_currentTargetDocumentId, true);
        }

        private void OnDocumentLoaded(Guid documentId)
        {
            if (_currentTargetDocumentId == Guid.Empty || documentId != _currentTargetDocumentId)
                return;
            _currentTargetDocumentId = Guid.Empty;

            Application.Current.Dispatcher.BeginInvoke((Action)(() =>
            {
                LayoutHelper.FindChildElement<Button>(Application.Current.MainWindow, x =>
                {
                    var dependencyObject = (DependencyObject)x;
                    return dependencyObject != null 
                    && dependencyObject.GetValue(AutomationProperties.AutomationIdProperty) as string == "showHideButton";
                })?.Command.Execute(null);

                OpenTargetInNewTab();
            }), DispatcherPriority.Background);
        }
    }
}
