/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;

namespace Ascon.Pilot.SDK.SearchSample.SourceFileSearch
{
    class FileSearchViewModel : IObserver<IDataObject>, INotifyPropertyChanged
    {
        private readonly IObjectsRepository _repository;
        private readonly ISearchService _searchService;
        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly Guid _searchContextFolderId;

        private string _searchString;
        private const int DefaultMaxResults = 20;
        private int _maxResults;
        private readonly string _searchContextFolderPath;
        private readonly ObservableCollection<FileItem> _fileItems = new ObservableCollection<FileItem>();
        private readonly HashSet<Guid> _loadedIds = new HashSet<Guid>();

        private bool _isBusy;
        private long? _total;

        private SearchResultsObserver _observer;
        private FileItem _selectedItem;

        public FileSearchViewModel(IObjectsRepository repository, ISearchService searchService, IStorageDataObject searchContextFolder, ITabServiceProvider tabServiceProvider)
        {
            _repository = repository;
            _searchService = searchService;
            _tabServiceProvider = tabServiceProvider;
            _searchContextFolderPath = searchContextFolder.Path;
            _searchContextFolderId = searchContextFolder.DataObject.Id;
        }

        public string SearchString
        {
            get { return _searchString; }
            set
            {
                _searchString = value;
                OnPropertyChanged("SearchString");
            }
        }

        public string SearchContext
        {
            get
            {
                return _searchContextFolderId != Guid.Empty
                    ? "поиск в папке " + _searchContextFolderPath
                    : "поиск по всей базе";
            }
        }

        public ObservableCollection<FileItem> FileItems
        {
            get { return _fileItems; }
        }

        public bool IsBusy
        {
            get { return _isBusy; }
            set
            {
                _isBusy = value;
                OnPropertyChanged("IsBusy");
            }
        }

        public string TotalDisplay
        {
            get { return "Найдено файлов: " + _total + "; Загружено результатов поиска: " + _loadedIds.Count; }
        }

        public bool HasSearch
        {
            get { return _total.HasValue; }
        }

        public void Search()
        {
            _total = null;
            _loadedIds.Clear();
            _fileItems.Clear();

            StartSearch(DefaultMaxResults);
        }

        public void LoadMore()
        {
            StartSearch(2 * _maxResults);
        }

        public void ShowFileOnDisk(Guid fileId)
        {
            _tabServiceProvider.ShowElement(fileId);
        }

        private void StartSearch(int maxResults)
        {
            IsBusy = true;
            _maxResults = maxResults;
            var queryBuilder = GetQueryBuilder(SearchString, _searchContextFolderId, _maxResults);
            _searchService.Search(queryBuilder).Subscribe(SearchResultsObserver);
        }

        private void OnSearchResultsLoaded(object sender, SearchResultsEventArgs e)
        {
            if (e.Kind == SearchResultKind.Local)
                return;

            _total = e.Total;
            var currentItems = _fileItems.Select(x => x.Id);
            var newItems = e.Ids.Except(currentItems).ToList();

            _repository.SubscribeObjects(newItems).Subscribe(this);
            OnPropertyChanged("TotalDisplay");
            OnPropertyChanged("HasSearch");

            IsBusy = false;
        }

        private IQueryBuilder GetQueryBuilder(string searchString, Guid contextId, int maxResults)
        {
            var builder = _searchService.GetEmptyQueryBuilder();
            builder.Must(ObjectFields.TypeId.Be(FileTypeId));
            builder.Must(ObjectFields.ObjectState.Be((int)ObjectState.Alive));

            if (!string.IsNullOrEmpty(searchString))
                builder.Must(ObjectFields.AllText.Be(searchString));

            builder.MaxResults(maxResults);
            builder.InContext(contextId);

            return builder;
        }

        private int FileTypeId
        {
            get { return _repository.GetType(SystemTypeNames.PROJECT_FILE).Id; }
        }

        private SearchResultsObserver SearchResultsObserver
        {
            get
            {
                if (_observer != null)
                    _observer.SearchResultsLoaded -= OnSearchResultsLoaded;

                _observer = new SearchResultsObserver();
                _observer.SearchResultsLoaded += OnSearchResultsLoaded;
                return _observer;
            }
        }

        public FileItem SelectedItem
        {
            get { return _selectedItem; }
            set
            {
                _selectedItem = value;
                OnPropertyChanged("SelectedItem");
            }
        }

        #region IObserver<IDataObject>
        public void OnNext(IDataObject value)
        {
            if (_loadedIds.Contains(value.Id))
                return;

            if (value.State == DataState.Loaded)
            {
                var path = _repository.GetStoragePath(value.Id);

                _fileItems.Add(string.IsNullOrEmpty(path)
                    ? new FileItem(value.Id, null, value.DisplayName)
                    : new FileItem(value.Id, path));

                _loadedIds.Add(value.Id);
                OnPropertyChanged("TotalDisplay");
                OnPropertyChanged("HasSearch");
            }
        }

        public void OnError(Exception error)
        {

        }

        public void OnCompleted()
        {

        }
        #endregion

        #region Property Changed
        public event PropertyChangedEventHandler PropertyChanged;

        protected virtual void OnPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
                PropertyChanged.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
        #endregion
    }

    class FileItem : INotifyPropertyChanged
    {
        public FileItem(Guid id, string path, string name = null)
        {
            Id = id;
            Path = path;
            Name = name ?? (path != null ? System.IO.Path.GetFileName(Path) : null);
        }

        public Guid Id { get; private set; }
        public string Path { get; private set; }
        public string Name { get; private set; }

        public override bool Equals(object obj)
        {
            var other = obj as FileItem;
            return other != null && other.Id == Id;
        }

        public override int GetHashCode()
        {
            return Id.GetHashCode();
        }

        #region Property Changed
        public event PropertyChangedEventHandler PropertyChanged;

        protected virtual void OnPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
                PropertyChanged.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
        #endregion
    }
}
