/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Windows.Media;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.WPFSample
{
    [Export(typeof(INewTabPage)), Export(typeof(ITabStateHandler))]
    public class OpenSpacePlugin : INewTabPage, IHandle<UnloadedEventArgs>, IHandle<CloseTabEventArgs>, ITabStateHandler
    {
        private static Guid OpenSpacePluginStateSerializationId => Guid.Parse("6ea31145-43b0-4339-a30a-5cc91daf6921");

        private readonly Dictionary<Guid, MainView> _tabsOpenByExtension = new Dictionary<Guid, MainView>();

        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly IEventAggregator _eventAggregator;
        private const string NewWPFExtensionTabCommandName = "WPFControlsExtension";
        private const string Title = "WPF controls extension";

        private const string SingleCommandName = "SingleWPFControlsExtension";
        private const string SingleTitle = "Single WPF controls extension";

        private const string OpenInOtherPaneCommandName = "WPFControlsExtensionInOtherPane";
        private const string OpenInOtherPaneTitle = "Open WPF controls extension in other pane";

        private const string CloseThisCommandName = "WPFControlsExtensionClose";
        private const string CloseThisTitle = "Close this tab page extension";

        private const string OpenNewTabCommandName = "WPFControlsExtensionOpenNewTab";
        private const string OpenNewTabTitle = "Open new tab page extension";

        [ImportingConstructor]
        public OpenSpacePlugin(ITabServiceProvider tabServiceProvider, IPilotDialogService dialogService, IEventAggregator eventAggregator)
        {
            _tabServiceProvider = tabServiceProvider;
            _eventAggregator = eventAggregator;
            _eventAggregator.Subscribe(this);
            var accentColor = (Color)ColorConverter.ConvertFromString(dialogService.AccentColor);
            ColorScheme.Initialize(accentColor, dialogService.Theme);
        }

        public void BuildNewTabPage(INewTabPageHost host)
        {
            var toolTip = "Open tab with WPF Controls extension";
            var icon = IconLoader.GetIcon("plugin_icon.svg");
            var groupId = Guid.Parse("70E21FDC-8676-47C5-887C-AD4872525717");
            host.SetGroup("WPF extension sample", groupId);
            host.AddButtonToGroup(Title, NewWPFExtensionTabCommandName, toolTip, icon, groupId);
            toolTip = "Open tab with single WPF controls extension";
            host.AddButtonToGroup(SingleTitle, SingleCommandName, toolTip, icon, groupId);
            toolTip = "Open WPF controls extension in other pane";
            host.AddButtonToGroup(OpenInOtherPaneTitle, OpenInOtherPaneCommandName, toolTip, icon, groupId);
            toolTip = "Open new tab page extension";
            host.AddButtonToGroup(OpenNewTabTitle, OpenNewTabCommandName, toolTip, icon, groupId);
            toolTip = "Close this tab page extension";
            host.AddButtonToGroup(CloseThisTitle, CloseThisCommandName, toolTip, icon, groupId);
        }

        public void OnButtonClick(string name)
        {
            var activeTabPageTitle = _tabServiceProvider.GetActiveTabPageTitle();

            if (name == NewWPFExtensionTabCommandName)
            {
                var view = new MainView();
                _tabServiceProvider.UpdateTabPageContent(activeTabPageTitle, Title, view);
                _tabsOpenByExtension.Add(_tabServiceProvider.GetActiveTabPageId(), view);
                return;
            }

            if (name == SingleCommandName)
            {
                OpenSingleExtensionTabPage(activeTabPageTitle);
                return;
            }

            if (name == OpenInOtherPaneCommandName)
            {
                var view = new MainView();
                var tabId = _tabServiceProvider.OpenTabPageAndGetId(OpenInOtherPaneTitle, view, true);
                _tabsOpenByExtension.Add(tabId, view);
                return;
            }

            if (name == CloseThisCommandName)
            {
                var activeTabPage = _tabServiceProvider.GetActiveTabPageTitle();
                _tabServiceProvider.CloseTabPage(activeTabPage);
                return;
            }

            if (name == OpenNewTabCommandName)
            {
                _tabServiceProvider.OpenNewTabPage();
            }
        }

        private void OpenSingleExtensionTabPage(string activeTabPageTitle)
        {
            var titles = _tabServiceProvider.GetTabPageTitles();
            if (titles.Contains(SingleTitle))
            {
                //close new tab page
                var activeTabPage = _tabServiceProvider.GetActiveTabPageTitle();
                _tabServiceProvider.CloseTabPage(activeTabPage);
                //activate single extension tab page
                _tabServiceProvider.ActivateTabPage(SingleTitle);
                return;
            }

            var singleView = new MainView();
            _tabServiceProvider.UpdateTabPageContent(activeTabPageTitle, SingleTitle, singleView);
            _tabsOpenByExtension.Add(_tabServiceProvider.GetActiveTabPageId(), singleView);
        }

        public void Handle(UnloadedEventArgs message)
        {
            _tabsOpenByExtension.Clear();
            _eventAggregator.Unsubscribe(this);
        }

        public void Handle(CloseTabEventArgs message)
        {
            if (_tabsOpenByExtension.TryGetValue(message.TabId, out var viewModel))
            {
                _tabsOpenByExtension.Remove(message.TabId);
            }
        }

        public bool TrySaveTabState(Guid tabId, out Guid extensionId, out string state, out string title)
        {
            var isOpenSpacePluginTab = _tabsOpenByExtension.TryGetValue(tabId, out var extensionTabView);
            if (!isOpenSpacePluginTab)
            {
                extensionId = Guid.Empty;
                state = null;
                title = null;
                return false;
            }

            extensionId = OpenSpacePluginStateSerializationId;
            state = extensionTabView.tabControl.SelectedIndex.ToString();
            title = Title;

            return true;
        }

        public bool TryAddTabAndRestoreState(Guid extensionId, string state, Guid tabId)
        {
            if (extensionId != OpenSpacePluginStateSerializationId)
                return false;

            var view = new MainView();
            _tabServiceProvider.UpdateTabPageContent(tabId, Title, view);
            _tabsOpenByExtension.Add(tabId, view);

            if (int.TryParse(state, out var savedTabIndex))
            {
                view.tabControl.SelectedIndex = savedTabIndex;
            }

            return true;
        }
    }
}