/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.IO;
using System.Threading.Tasks;
using System.Timers;
using System.Windows.Media;
using System.Windows.Media.Imaging;

namespace Ascon.Pilot.SDK.XpsToBitmapRendererSample
{
    public class RenderedPagesViewModel : INotifyPropertyChanged, IDisposable
    {
        private readonly IXpsRender _render;
        private readonly Stream _xpsStream;
        private readonly Timer _elaspedTimer;

        public event PropertyChangedEventHandler PropertyChanged;

        private Collection<ImageSource> _pageCollection;
        public Collection<ImageSource> PagesCollection
        {
            get { return _pageCollection; }
            set
            {
                _pageCollection = value;
                OnPropertyChanged("PagesCollection");
            }
        }

        public RenderedPagesViewModel(IXpsRender render, Stream xpsStream)
        {
            _render = render;
            _xpsStream = xpsStream;

            _elaspedTimer = new Timer(100);
            _elaspedTimer.Elapsed += (sender, args) => Init();
            _elaspedTimer.Start();
        }

        public void Init()
        {
            _elaspedTimer.Stop();

            new Task(() =>
            {
                var previews = _render.RenderXpsToBitmap(_xpsStream);
                var pages = new Collection<ImageSource>();

                foreach (var preview in previews)
                {
                    preview.Position = 0;

                    var bitmap = new BitmapImage();
                    bitmap.BeginInit();
                    bitmap.StreamSource = preview;
                    bitmap.CacheOption = BitmapCacheOption.OnLoad;
                    bitmap.EndInit();
                    bitmap.Freeze();

                    pages.Add(bitmap);
                }
                PagesCollection = pages;
            }).Start();
        }

        public void Dispose()
        {
            if (_elaspedTimer != null)
                _elaspedTimer.Dispose();
            if (_xpsStream != null)
                _xpsStream.Dispose();
        }

        protected virtual void OnPropertyChanged(string propertyName)
        {
            var handler = PropertyChanged;
            if (handler != null) 
                handler(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}