/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Interop;
using System.Xml.Linq;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.XpsViewerSample.Properties;


namespace Ascon.Pilot.SDK.XpsViewerSample
{
    [Export(typeof(IMenu<GraphicLayerElementContext>))]
    public class GraphicLayerContextMenuSample : IMenu<GraphicLayerElementContext>
    {
        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;
        private readonly IFileProvider _fileProvider;

        [ImportingConstructor]
        public GraphicLayerContextMenuSample(IObjectModifier modifier, IObjectsRepository repository, IFileProvider fileProvider)
        {
            _modifier = modifier;
            _repository = repository;
            _fileProvider = fileProvider;
        }

        public void Build(IMenuBuilder builder, GraphicLayerElementContext context)
        {
            builder.AddItem("miChangeTextInGraphicLayerElement", builder.Count).WithHeader(Resources.EditGraphicLayerElementTitle);
            builder.AddItem("miChangeGraphicLayerElement", builder.Count).WithHeader(Resources.EditGraphicLayerElementDescription);
        }

        public void OnMenuItemClick(string name, GraphicLayerElementContext context)
        {
            if (name == "miChangeTextInGraphicLayerElement")
            {
                ChangeGraphicLayerElement(context.DataObject, context.ContentId, GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT_CONTENT);
            }

            if (name == "miChangeGraphicLayerElement")
            {
                ChangeGraphicLayerElement(context.DataObject, context.ElementId, GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT);
            }
        }

        [DllImport("user32.dll")]
        public static extern IntPtr GetForegroundWindow();

        private void ChangeGraphicLayerElement(IDataObject dataObject, Guid elementId, string graphicElementType)
        {
            var graphicLayerContent = dataObject.Files.FirstOrDefault(f => f.Name.Contains(elementId.ToString()));
            if (graphicLayerContent != null)
            {
                XElement xamlObject;
                if (!TryParseXamlObject(graphicLayerContent, out xamlObject))
                    return;

                var title = Resources.EditGraphicLayerElementTitle;
                if (graphicElementType == GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT)
                    title = Resources.EditGraphicLayerElementDescription;

                var builder = _modifier.Edit(dataObject);
                var viewModel = new EditDialogViewModel(xamlObject.ToString(), title);
                var editView = new EditDialogView { DataContext = viewModel };
                var activeWindowHandle = GetForegroundWindow();
                new WindowInteropHelper(editView)
                {
                    Owner = activeWindowHandle
                };

                if (editView.ShowDialog() == true)
                {
                    xamlObject = XElement.Parse(viewModel.Text);
                    using (var textBlocksStream = new MemoryStream())
                    {
                        using (var writer = new StreamWriter(textBlocksStream))
                        {
                            writer.Write(xamlObject);
                            writer.Flush();
                            builder.AddOrReplaceFile(graphicElementType + elementId, textBlocksStream, graphicLayerContent, DateTime.Now, DateTime.Now, DateTime.Now);
                        }
                    }
                    _modifier.Apply();
                }
            }
        }

        private bool TryParseXamlObject(IFile graphicLayerElement, out XElement xamlObject)
        {
            xamlObject = null;
            using (var stream = _fileProvider.OpenRead(graphicLayerElement))
            {
                using (var sr = new StreamReader(stream))
                {
                    try
                    {
                        xamlObject = XElement.Parse(sr.ReadToEnd());
                    }
                    catch (System.Xml.XmlException)
                    {
                        MessageBox.Show(Resources.CannotEditElement);
                        return false;
                    }
                }
            }
            return true;
        }
    }
}