/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Windows;
using System.Windows.Input;
using System.Xml.Serialization;
using Ascon.Pilot.SDK.GraphicLayerSample;
using Ascon.Pilot.SDK.HotKey;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.XpsViewerSample.Properties;

namespace Ascon.Pilot.SDK.XpsViewerSample
{
    [Export(typeof(IMenu<XpsRenderClickPointContext>)), Export(typeof(IHotKey<XpsRenderContext>))]
    public class XpsRenderContextMenuSample : IMenu<XpsRenderClickPointContext>, IHotKey<XpsRenderContext>
    {
        private readonly IObjectModifier _modifier;
        private readonly IPerson _currentPerson;
        private const string CreateElement = "mCreateGraphicLayerElementAtPoint";

        [ImportingConstructor]
        public XpsRenderContextMenuSample(IObjectModifier modifier, IObjectsRepository repository)
        {
            _modifier = modifier;
            _currentPerson = repository.GetCurrentPerson();
        }

        public void Build(IMenuBuilder builder, XpsRenderClickPointContext context)
        {
            var items = builder.ItemNames.ToList();
            var position = items.Contains("miAddGraphicAnnotation") ? items.IndexOf("miAddGraphicAnnotation") : items.Count - 1;
            builder.AddItem(CreateElement, position + 1)
                   .WithHeader(Resources.CreateGraphicLayerElementAtPoint)
                   .WithIsEnabled(context.SelectedVersion == context.DataObject.ActualFileSnapshot.Created);
        }

        public void OnMenuItemClick(string name, XpsRenderClickPointContext context)
        {
            if (name == CreateElement)
            {
                CreateGraphicLayerElementAtPoint(context);
            }
        }

        private void CreateGraphicLayerElementAtPoint(XpsRenderClickPointContext context)
        {
            var elementId = Guid.NewGuid();
            var positionId = _currentPerson.MainPosition.Position;
            var xamlContent = GraphicLayerElementCreator.CreateStampWithDateTime().ToString();
            var elementName = GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT + elementId + "_" + positionId;
            var element = CreateGraphicLayerElement(context.ClickPoint, context.PageNumber, elementId, positionId);

            var builder = _modifier.Edit(context.DataObject);
            using (var graphicElementStream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(GraphicLayerElement));
                serializer.Serialize(graphicElementStream, element);
                graphicElementStream.Position = 0;
                builder.AddFile(elementName, graphicElementStream, DateTime.Now, DateTime.Now, DateTime.Now);
            }

            using (var contentStream = new MemoryStream())
            {
                using (var writer = new StreamWriter(contentStream))
                {
                    writer.Write(xamlContent);
                    writer.Flush();
                    builder.AddFile(GraphicLayerElementConstants.GRAPHIC_LAYER_ELEMENT_CONTENT + element.ContentId,
                        contentStream, DateTime.Now, DateTime.Now, DateTime.Now);
                }
            }
            _modifier.Apply();
        }

        private GraphicLayerElement CreateGraphicLayerElement(Point clickPoint, int pageNumber, Guid elementId, int positionId)
        {
            var scale = new Point(1, 1);
            return new GraphicLayerElement(elementId, Guid.NewGuid(), clickPoint.X, clickPoint.Y, positionId, scale,
                0, VerticalAlignment.Top, HorizontalAlignment.Left, GraphicLayerElementConstants.XAML, pageNumber, true);
        }

        #region IHotKey implementation
        private const string PRINT_COMMAND_ID = "sampleExtensionPrintCommand";
        private const string SHOW_INFO_COMMAND_ID = "sampleExtensionShowInfoCommand";
        public void AssignHotKeys(IHotKeyCollection hotKeyCollection)
        {
            hotKeyCollection.Add(Key.P, ModifierKeys.Control, PRINT_COMMAND_ID);//overrides default hot key
            hotKeyCollection.Add(Key.I, ModifierKeys.Control, SHOW_INFO_COMMAND_ID);//adds new hot key
        }

        public void OnHotKeyPressed(string commandId, XpsRenderContext context)
        {
            switch (commandId)
            {
                case PRINT_COMMAND_ID:
                {
                    MessageBox.Show("Default print document routine override logic here.", "XPS Viewer Hot key extension sample.");
                    break;
                }
                case SHOW_INFO_COMMAND_ID:
                {
                    MessageBox.Show(context.DataObject.DisplayName, "XPS Viewer Hot key extension sample.");
                    break;
                }
            }
        }
        #endregion
    }
}