﻿using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.Tools;
using System;
using System.Globalization;
using System.Linq;

namespace AttributeHistoryBot
{
    class AttrDiff
    {
        public AttrDiff(string oldValue, string newValue)
        {
            OldValue = oldValue;
            NewValue = newValue;
        }

        public string OldValue { get; }
        public string NewValue { get; }
    }

    class AttrDiffProvider
    {
        public static AttrDiff CalcDiff(object oldValue, object newValue, IAttribute attribute, IAttributeFormatParser parser, IObjectsRepository repository)
        {
            if (Equals(oldValue, newValue))
                return null;

            switch (attribute.Type)
            {
                case AttributeType.String:
                    return new AttrDiff((string)oldValue, (string)newValue);
                case AttributeType.Numerator:
                    return new AttrDiff(null, (string)newValue);
                case AttributeType.UserState:
                    return new AttrDiff(GetStateTitle(oldValue, repository), GetStateTitle(newValue, repository));
                case AttributeType.DateTime:
                case AttributeType.Integer:
                case AttributeType.Double:
                case AttributeType.Decimal:
                    var formatter = new Formatter(attribute, parser);
                    var oldFormattedValue = formatter.Format(oldValue);
                    var newFormattedValue = formatter.Format(newValue);
                    return new AttrDiff(oldFormattedValue, newFormattedValue);
                case AttributeType.Boolean:
                    var oldBooleanValue = oldValue as bool? ?? false;
                    var newBooleanValue = newValue as bool? ?? false;
                    return new AttrDiff(FormatBool(oldBooleanValue), FormatBool(newBooleanValue));
                case AttributeType.OrgUnit:
                    var oldOrgUnitIds = oldValue as int[];
                    var newOrgUnitIds = newValue as int[];
                    var oldOrgUnitsStr = FormatOrgUnits(oldOrgUnitIds, repository);
                    var newOrgUnitsStr = FormatOrgUnits(newOrgUnitIds, repository);
                    return new AttrDiff(oldOrgUnitsStr, newOrgUnitsStr);
            }   
            return null;
        }

        private static string GetStateTitle(object value, IObjectsRepository repository)
        {
            if (!(value is Guid stateId))
                return null;

            var state = repository.GetUserStates().FirstOrDefault(x => x.Id == stateId);
            return state?.Title;
        }

        private static string FormatOrgUnits(int[] ids, IObjectsRepository repository)
        {
            return ids?.Length > 0
                ? string.Join(",", ids.Select(id => GetPersonOnOrgUnitTitle(id, repository)))
                : null;
        }

        private static string GetPersonOnOrgUnitTitle(int id, IObjectsRepository repository)
        {
            var orgUnit = repository.GetOrganisationUnit(id);
            var personOnOrgUnit = repository.GetPersonOnOrganizationUnit(id);
            return orgUnit.GetTitle(personOnOrgUnit);
        }

        private static string FormatBool(bool b)
        {
            if (b)
                return "[v]";
            return "[ ]";
        }

        class Formatter
        {
            private readonly CultureInfo _culture;
            private readonly string _format;

            public Formatter(IAttribute attribute, IAttributeFormatParser parser)
            {
                _format = parser.GetAttributeFormat(attribute.Configuration2());
                if(string.IsNullOrEmpty(_format) && attribute.Type == AttributeType.DateTime)
                    _format = CultureInfo.CurrentCulture.DateTimeFormat.ShortDatePattern;

                var cultureName = parser.GetAttributeFormatCulture(attribute.Configuration2());
                _culture = !string.IsNullOrEmpty(cultureName)
                    ? new CultureInfo(cultureName)
                    : CultureInfo.CurrentUICulture;
            }

            public string Format(object value)
            {
                if (string.IsNullOrEmpty(_format) || !(value is IFormattable formattable))
                    return value?.ToString();

                return formattable.ToString(_format, _culture);
            }
        }
    }
}
