/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Commands;
using Ascon.Pilot.SDK.Menu;

namespace CommandInvokerSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    class ContextMenuImpl : IMenu<ObjectsViewContext>
    {
        private readonly ICommandInvoker _commandInvoker;
        private readonly IObjectModifier _objectModifier;
        private readonly IFileSaver _fileSaver;

        // Command Id for converting pdf to xps, will work only with "PDF2XPS" extension and it's active license
        private readonly Guid _convertPdfToXpsCommandId = new Guid("AFDEAC4F-3DA8-4CEB-B77B-2C956B7923B4");

        [ImportingConstructor]
        public ContextMenuImpl(ICommandInvoker commandInvoker, IObjectModifier objectModifier, IFileSaver fileSaver)
        {
            _commandInvoker = commandInvoker;
            _objectModifier = objectModifier;
            _fileSaver = fileSaver;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            builder.AddItem("CallFirstCommand", 0)
                .WithHeader("Call first command from CommandHandlerSample extension");

            builder.AddItem("CallSecondCommand", 1)
                .WithHeader("Call second command from CommandHandlerSample extension with params");

            builder.AddItem("CallSecondCommand2", 2)
                .WithHeader("Call second command from CommandHandlerSample extension without params");

            if (context.SelectedObjects.Any(x => x.Files.Any(f => f.Name.EndsWith("pdf"))))
            {
                builder.AddItem("ConvertPdfToXpsCommand", 3)
                    .WithHeader("Convert pdf to xps");
            }
        }

        public async void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name == "CallFirstCommand")
            {
                var commandId = new Guid("2039E6FB-46AF-4CF8-B422-45E589FFF6ED");
                await InvokeCommand(commandId, context, "Param1");
                return;
            }

            if (name == "CallSecondCommand")
            {
                var commandId = new Guid("BDF59966-590A-47BE-802D-7740CDCC1C5B");
                await InvokeCommand(commandId, context, "Param 2");
                return;
            }

            if (name == "CallSecondCommand2")
            {
                var commandId = new Guid("BDF59966-590A-47BE-802D-7740CDCC1C5B");
                await InvokeCommand(commandId, context, null);
            }

            if (name == "ConvertPdfToXpsCommand")
            {
                await InvokeCommand(_convertPdfToXpsCommandId, context, null);
            }

        }

        private async Task InvokeCommand(Guid commandId, ObjectsViewContext context, string param)
        {
            try
            {
                byte[] bytes;

                if (commandId == _convertPdfToXpsCommandId)
                {
                    var objectsWithPdf = context.SelectedObjects.Where(o => o.ActualFileSnapshot.Files.Any(f => f.Name.EndsWith("pdf")));
                    await SavePdfAndConvertToXpsAsync(objectsWithPdf);
                    return;
                }
                if (param == null)
                {
                    bytes = _commandInvoker.Invoke(commandId, null);
                }
                else
                {
                    var args = Encoding.Unicode.GetBytes(param);
                    bytes = _commandInvoker.Invoke(commandId, args);
                }

                var messageFromOtherExtension = Encoding.Unicode.GetString(bytes);
                MessageBox.Show(messageFromOtherExtension);
            }
            catch (CommandHandlerNotFoundException e)
            {
                MessageBox.Show($"Command handler for command {e.CommandId} not found", "CommandInvokerSample");
            }

        }

        private async Task SavePdfAndConvertToXpsAsync(IEnumerable<IDataObject> objectsWithPdf)
        {
            var filesPaths = new List<string>();
            foreach (var obj in objectsWithPdf)
            {
                var pdf = obj.Files.FirstOrDefault(x => x.Name.EndsWith("pdf"));
                if (pdf == null) continue;
                var outputFilePath = Path.GetTempPath() + $"{pdf.Name}";
                await _fileSaver.SaveFileAsync(obj, outputFilePath);

                string destPath;
                var resultBytes = await Task.Run(() => _commandInvoker.Invoke(_convertPdfToXpsCommandId, Encoding.Unicode.GetBytes(outputFilePath)));
                if (resultBytes != null)
                {
                    var sourceXpsFile = Encoding.Unicode.GetString(resultBytes);
                    destPath = Path.ChangeExtension(outputFilePath, ".xps");

                    File.Move(sourceXpsFile, destPath);
                    filesPaths.Add(destPath);
                    File.Delete(sourceXpsFile);
                }
                else
                    throw new Exception("There was an error while converting pdf to xps");

                ReplacePdfWithXps(obj, destPath);
            }

            _objectModifier.Apply();
            filesPaths.ForEach(File.Delete);
        }

        private void ReplacePdfWithXps(IDataObject obj, string fileForNewSnapShot)
        {
            var objBuilder = _objectModifier
                                        .Edit(obj)
                                        .CreateFileSnapshot("Replace pdf to xps");
            objBuilder.AddFile(fileForNewSnapShot);
        }
    }
}
