﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Org.BouncyCastle.Security;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;
using System.Text.RegularExpressions;
using Newtonsoft.Json;
using Org.BouncyCastle.Utilities.Encoders;
using Ascon.Pilot.SDK;

namespace CryptoProvider.CryptoAPI
{
    [Export(typeof(ICryptoProvider))]
    [Export(typeof(IAutomationSigner))]
    public class CryptoProvider : ICryptoProvider, IAutomationSigner, IObserver<KeyValuePair<string, string>>
    {
        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;
        private readonly IObjectsRepository _repository;
        private readonly Regex _cnRegex = new Regex(@"(?:CN|cn)=([^\n,]+)");

        private readonly List<string> _certificateEnhancementSettings = new List<string>();

        private const string SIG_EXTENSION = "sig";
        Lazy<HashSet<string>> _supportedAlgorithms = new Lazy<HashSet<string>>(AlgorithmsFinder.FindAllSupportedAlgorithms);

        [ImportingConstructor]
        public CryptoProvider(IObjectModifier modifier, IFileProvider fileProvider, IObjectsRepository repository, IPersonalSettings personalSettings)
        {
            _repository = repository;
            _modifier = modifier;
            _fileProvider = fileProvider;
            personalSettings.SubscribeSetting(SystemSettingsKeys.AdvancedSignatureFeatureKey).Subscribe(this);
        }

        public IEnumerable<ICertificate> GetCertificates()
        {
            var collection = GetCertificatesCollection();
            foreach (var cert in collection)
            {
                if (_supportedAlgorithms.Value.Contains(cert.PublicKey.Oid.Value))
                    yield return new Certificate(cert);
            }
        }
        public void Sign(IObjectModifier modifier, Guid documentId, IFile file, Stream stream, ICertificate cert, IEnumerable<ISignatureRequest> signatureRequests)
        {
            SignCore(modifier, documentId, file, stream, cert, signatureRequests);
        }
        public void Sign(Guid documentId, IFile file, Stream stream, ICertificate cert,
            IEnumerable<ISignatureRequest> signatureRequests)
        {
            _modifier.Clear();
            SignCore(_modifier, documentId, file, stream, cert, signatureRequests);
            _modifier.Apply();
        }
        public bool IsAlgorithmSupported(string algorithmOid)
        {
            return _supportedAlgorithms.Value.Any(x => x == algorithmOid);
        }

        public bool IsAlgorithmSupported(byte[] sigBase64)
        {
            try
            {
                var signatureData = Base64.Decode(sigBase64);
                var cert = new X509Certificate2(signatureData);
                var algorithm = cert.PublicKey.Oid.Value.ToString();
                return _supportedAlgorithms.Value.Any(x => x == algorithm);
            }
            catch (Exception)
            {
                return false;
            }
        }

        public void ReadImportedSignature(Stream file, byte[] sigBase64, IImportedSignatureListener listener)
        {
            try
            {
                var signatureData = Base64.Decode(sigBase64);
                var isSelfSigned = IsCertificateSelfSigned(signatureData);
                var cadesType = Asn1Helper.GetCadesType(signatureData, isSelfSigned);

                var documentData = StreamToArray(file);
                ContentInfo contentInfo = new ContentInfo(documentData);
                SignedCms cms = new SignedCms(contentInfo, true);
                cms.Decode(signatureData);

                cms.CheckSignature(true);

                var signer = cms.SignerInfos[0];
                if (cadesType >= CadesType.CadesT)
                    CheckTimeStampSignature(signer);

                listener.SignatureIsValid();

                if (CheckCertificate(signer.Certificate))
                    listener.CertificateIsValid();

                listener.SetCadesType(cadesType);

                var cert = new X509Certificate2(signatureData);
                listener.SetPublicKeyOid(cert.PublicKey.Oid.Value);
            }
            catch (Exception e)
            {
                listener.SignatureIsInvalid(e.Message);
            }
            finally
            {
                listener.ReadCompleted();
            }

        }


        public void ReadSignature(IDataObject document, Stream stream, IFilesSnapshot snapshot,
            ISignatureRequest signatureRequest,
            IReadSignatureListener listener)
        {
            if (!signatureRequest.Signs.Any() || !Guid.TryParse(signatureRequest.Signs.Last(), out var signId))
                return;

            var signFile = snapshot.Files.FirstOrDefault(x => x.Id == signId);
            if (signFile == null)
                return;

            try
            {
                using (var signatureStream = _fileProvider.OpenRead(signFile))
                {
                    var signatureData = ConvertFromBase64(signatureStream, signatureRequest.PublicKeyOid);
                    var orgUnit = _repository.GetOrganisationUnit(signatureRequest.PositionId);
                    var documentData = StreamToArray(stream);
                    ContentInfo contentInfo = new ContentInfo(documentData);
                    SignedCms cms = new SignedCms(contentInfo, true);
                    cms.Decode(signatureData);
                    var signer = cms.SignerInfos[0];

                    var subjectCn = GetCNPart(signer.Certificate.Subject);
                    var signerName = string.IsNullOrEmpty(orgUnit.Title) ? subjectCn : $"{subjectCn} ({orgUnit.Title})";
                    listener.SetSignerName(signerName);

                    string signDate = signatureRequest.LastSignCadesType == CadesType.CadesBes ? GetSignDate(signer) : GetSignDateFromTimeStamp(signer);
                    if (!string.IsNullOrEmpty(signDate))
                        listener.SetSignDate(signDate);
                    else
                        listener.SetSignDate(Properties.Resources.UnknownDateTime);

                    cms.CheckSignature(true);
                    if (signatureRequest.LastSignCadesType >= CadesType.CadesT)
                        CheckTimeStampSignature(signer);

                    listener.SignatureIsValid();

                    if (CheckCertificate(signer.Certificate))
                        listener.CertificateIsValid();
                }

            }
            catch (Exception e)
            {
                listener.SignatureIsInvalid(e.Message);
            }
            finally
            {
                listener.ReadCompleted();
            }
        }

        private void SignCore(IObjectModifier modifier, Guid documentId, IFile file, Stream stream, ICertificate cert, IEnumerable<ISignatureRequest> signatureRequests)
        {
            var builder = modifier.EditById(documentId);

            var documentData = StreamToArray(stream);
            ContentInfo contentInfo = new ContentInfo(documentData);
            SignedCms cms = new SignedCms(contentInfo, true);

            X509Store store = new X509Store(StoreName.My, StoreLocation.CurrentUser);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            var certificate = store.Certificates.Find(X509FindType.FindByThumbprint, cert.Thumbprint, false)[0];

            CmsSigner signer = new CmsSigner(certificate);
            signer.SignedAttributes.Add(new Pkcs9SigningTime());
            cms.ComputeSignature(signer, false);
            var signatureDataArray = cms.Encode();
            var signatureData = Convert.ToBase64String(signatureDataArray);

            foreach (var request in signatureRequests)
            {
                using (var ms = new MemoryStream())
                using (var writer = new StreamWriter(ms))
                {
                    writer.Write(signatureData);
                    writer.Flush();
                    ms.Position = 0;
                    builder.AddFile($"{file.Name}{request.Id}.{SIG_EXTENSION}", ms, DateTime.UtcNow,
                        DateTime.UtcNow, DateTime.UtcNow, out var signatureFileId);

                    var issuerCn = GetCNPart(signer.Certificate.Issuer);

                    var isAdvancementRequired = false;

                    if (!String.IsNullOrEmpty(issuerCn))
                    {
                        var formattedIssuer = issuerCn.Replace("\"", string.Empty);
                        if (_certificateEnhancementSettings.Contains(formattedIssuer))
                            isAdvancementRequired = true;
                    }

                    builder.SetSignatures(f => f.Id == file.Id).SetPublicKeyOid(request.Id, cert.PublicKeyOid).SetLastSignCadesType(request.Id, CadesType.CadesBes).SetIsAdvancementRequired(request.Id, isAdvancementRequired).SetSign(request.Id, signatureFileId.ToString());

                }
            }
        }
        private static byte[] StreamToArray(Stream stream)
        {
            using (var ms = new MemoryStream())
            {
                stream.CopyTo(ms);
                return ms.ToArray();
            }
        }

        private static X509Certificate2Collection GetCertificatesCollection()
        {
            var store = new X509Store(StoreLocation.CurrentUser);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            var certificateCollection = store.Certificates;
            store.Close();

            certificateCollection = certificateCollection.Find(X509FindType.FindByTimeValid, DateTime.Now, true);
            certificateCollection = certificateCollection.Find(X509FindType.FindByKeyUsage, X509KeyUsageFlags.DigitalSignature, false);
            var withPrivate = new X509Certificate2Collection();
            foreach (X509Certificate2 cer in certificateCollection.Cast<X509Certificate2>().Where(cer => cer.HasPrivateKey))
                withPrivate.Add(cer);
            certificateCollection = withPrivate;

            return certificateCollection;
        }

        private string GetCNPart(string str)
        {
            if (string.IsNullOrEmpty(str))
                return string.Empty;

            var matches = _cnRegex.Matches(str);
            if (matches.Count == 0)
                return string.Empty;

            return matches[0].Value.Split('=')[1];
        }


        private static bool IsCertificateSelfSigned(byte[] certificateData)
        {
            var parser = new Org.BouncyCastle.X509.X509CertificateParser();
            var cert = parser.ReadCertificate(certificateData);
            return cert?.SubjectDN?.Equivalent(cert?.IssuerDN) ?? false;
        }

        private void CheckTimeStampSignature(SignerInfo signer)
        {
            var token = Asn1Helper.GetTimestampToken(signer);
            var imprintDigest = token.TimeStampInfo.GetMessageImprintDigest();
            var signature = signer.GetSignature();
            var hash = DigestUtilities.CalculateDigest(token.TimeStampInfo.HashAlgorithm.Algorithm, signature);
            if (!Enumerable.SequenceEqual(imprintDigest, hash))
                throw new Exception("time stamp hash is different");

            var certEncoded = Asn1Helper.GetTimeStampCertificate(token);
            var certTs = new Org.BouncyCastle.X509.X509Certificate(certEncoded);
            token.Validate(certTs);
        }

        public static byte[] ConvertFromBase64(Stream stream, string publicKeyOid)
        {
            using (StreamReader reader = new StreamReader(stream))
            {
                string result = reader.ReadToEnd();
                return Convert.FromBase64String(result);
            }
        }

        private string GetSignDate(SignerInfo signerInfo)
        {
            var signDate = Asn1Helper.GetSignDate(signerInfo);
            if (signDate.HasValue)
                return signDate.Value.ToLocalTime().ToString("g");

            return string.Empty;
        }

        private string GetSignDateFromTimeStamp(SignerInfo signerInfo)
        {
            var token = Asn1Helper.GetTimestampToken(signerInfo);
            return token.TimeStampInfo.GenTime.ToLocalTime().ToString("g");
        }

        private bool CheckCertificate(X509Certificate certificate)
        {
            var x5092Cert = new X509Certificate2(certificate);
            var chain = new X509Chain
            {
                ChainPolicy =
                {
                    RevocationMode = X509RevocationMode.NoCheck,
                    VerificationFlags = X509VerificationFlags.AllowUnknownCertificateAuthority | X509VerificationFlags.IgnoreNotTimeValid
                }
            };
            var isSoftCertificateValidation = chain.Build(x5092Cert);
            var isCertificateValid = false;
            if (isSoftCertificateValidation)
            {
                chain.Reset();
                chain.ChainPolicy.RevocationMode = X509RevocationMode.Online;
                chain.ChainPolicy.VerificationFlags = X509VerificationFlags.NoFlag;
                isCertificateValid = chain.Build(x5092Cert);
            }
            return isSoftCertificateValidation && isCertificateValid;
        }

        public void OnNext(KeyValuePair<string, string> value)
        {
            if (value.Value == null)
            {
                _certificateEnhancementSettings.Clear();
                return;
            }

            try
            {
                var certSettingsArray = value.Value.Split(new[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries).Select(x => x.Trim().Replace("\"", string.Empty));

                _certificateEnhancementSettings.Clear();
                _certificateEnhancementSettings.AddRange(certSettingsArray);
            }
            catch (JsonException ex)
            {
                System.Windows.Forms.MessageBox.Show(ex.Message, "Invalid settings format.", System.Windows.Forms.MessageBoxButtons.OK, System.Windows.Forms.MessageBoxIcon.Error);
            }
        }

        public void OnError(Exception error)
        {
        }

        public void OnCompleted()
        {
        }

    }
}
