﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Linq;
using Ascon.Pilot.SDK;
using PdfStamper.Models;
using CryptoProvider.CryptoAPI;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;

namespace PdfStamper
{
    internal class CertificateReader
    {
        private readonly IFileProvider _fileProvider;

        public CertificateReader(IFileProvider fileProvider)
        {
            _fileProvider = fileProvider;
        }

        public IEnumerable<ISignatureRequest> GetSignatureRequests(IDataObject document, DateTime documentVersion)
        {
            var snapshot = Tools.GetFilesSnapshot(document, documentVersion);

            if (snapshot == null)
                return Array.Empty<ISignatureRequest>();

            var signatureRequests = snapshot.Files
                .SelectMany(x => x.SignatureRequests)
                .Where(y => y.Signs.Any())
                .Where(x => x.LastSignCadesType != CadesType.Unknown && x.LastSignCadesType != CadesType.NotCades);

            return signatureRequests;

        }

        public IEnumerable<CertificateInfo> GetInfosForStamps(IDataObject document, DateTime documentVersion)
        {
            var snapshot = Tools.GetFilesSnapshot(document, documentVersion);
            if (snapshot == null)
                return Array.Empty<CertificateInfo>();

            var signatureRequests = GetSignatureRequests(document, documentVersion).ToList();

            if (!signatureRequests.Any())
                return Array.Empty<CertificateInfo>();

            var certificateInfoList = new List<CertificateInfo>();
            foreach (var request in signatureRequests)
            {
                var signFileId = request.Signs.LastOrDefault();
                if (string.IsNullOrEmpty(signFileId))
                    continue;

                var signatureFile = snapshot.Files.FirstOrDefault(x => x.Id.Equals(Guid.Parse(signFileId)) && x.Name.EndsWith(Constants.SIG_EXTENSION));
                if (signatureFile == null)
                    continue;

                if (request.LastSignCadesType >= CadesType.CadesT)
                {
                    using (var docStream = _fileProvider.OpenRead(snapshot.Files.FirstOrDefault(x => x.Name.EndsWith(".pdf", StringComparison.OrdinalIgnoreCase))))
                    {
                        var certificateInfo = GetCadesTCertInfo(signatureFile, request, docStream);
                        certificateInfoList.Add(certificateInfo);
                    }
                }
                else
                {
                    using (var signatureStream = _fileProvider.OpenRead(signatureFile))
                    {
                        var signature = Tools.StreamToArray(signatureStream);
                        using (var certificate = new X509Certificate2(signature))
                        {
                            var certificateInfo = new CertificateInfo
                            {
                                Id = signFileId,
                                SignerRole = request.Role,
                                Subject = GetCN(certificate.Subject),
                                Thumbprint = certificate.Thumbprint,
                                ValidFrom = certificate.NotBefore.ToShortDateString(),
                                ValidTo = certificate.NotAfter.ToShortDateString()
                            };
                            certificateInfoList.Add(certificateInfo);
                        }
                    }
                }
            }

            return certificateInfoList;
        }

        private CertificateInfo GetCadesTCertInfo(IFile signFile, ISignatureRequest signatureRequest, Stream docStream)
        {
            SignerInfo signer;
            using (var signatureStream = _fileProvider.OpenRead(signFile))
            {
                var signatureData = ConvertFromBase64(signatureStream);
                var documentData = Tools.StreamToArray(docStream);
                var contentInfo = new ContentInfo(documentData);
                var cms = new SignedCms(contentInfo, true);
                cms.Decode(signatureData);
                signer = cms.SignerInfos[0];
            }

            var token = Asn1Helper.GetTimestampToken(signer);
            var certEncoded = Asn1Helper.GetTimeStampCertificate(token);
            var cert = new Org.BouncyCastle.X509.X509Certificate(certEncoded);
            return new CertificateInfo
            {
                Id = signatureRequest.Signs.LastOrDefault(),
                SignerRole = signatureRequest.Role,
                Subject = GetCN(signer.Certificate.Subject),
                Thumbprint = signer.Certificate.Thumbprint,
                ValidFrom = cert.NotBefore.ToShortDateString(),
                ValidTo = cert.NotAfter.ToShortDateString()
            };

        }

        public static byte[] ConvertFromBase64(Stream stream)
        {
            using (var reader = new StreamReader(stream))
            {
                var result = reader.ReadToEnd();
                return Convert.FromBase64String(result);
            }
        }

        private static string GetCN(string subject)
        {
            var match = Regex.Match(subject, @"CN=([^,]+)");
            return match.Success ? match.Groups[1].Value : subject;
        }
    }
}
