﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using PdfSharp.Pdf;
using PdfSharp.Fonts;
using PdfSharp.Pdf.IO;
using PdfSharp.Drawing;
using System.Reflection;
using PdfSharp.Pdf.Advanced;
using PdfSharp.Drawing.Layout;
using PdfSharp.Pdf.Annotations;
using Ascon.Pilot.SDK;
using PdfStamper.Models;
using PdfStamper.Resources;

namespace PdfStamper
{
    internal class StampCreator : IDisposable
    {
        private readonly PdfDocument _document;
        private readonly PdfStamperMode _stamperMode;
        private readonly MemoryStream _documentStream;

        public StampCreator(Stream originalDocumentStream, PdfStamperMode stamperMode)
        {
            GlobalFontSettings.FontResolver = new FontResolver();
            _stamperMode = stamperMode;
            _documentStream = new MemoryStream(Tools.StreamToArray(originalDocumentStream));
            _document = PdfReader.Open(_documentStream, PdfDocumentOpenMode.Modify);
        }

        public int GetPagesCount()
        {
            return _document.PageCount;
        }

        public XSize GetPageSize(int pageNumber)
        {
            if (_document.Pages.Count < pageNumber - 1)
                throw new ArgumentOutOfRangeException(nameof(pageNumber));
            
            var height = _document.Pages[pageNumber].Height;
            var width = _document.Pages[pageNumber].Width;

            return new XSize(width.Point, height.Point);
        }

        public void SaveStamps(Stream stream)
        {
            _document?.Save(stream);
        }

        public void AddSignatureStamp(int pageNumber, CertificateInfo certInfo, StampPosition stampPosition, XSize size, SignatureSettingModel model)
        {
            var page = _document.Pages[pageNumber];

            // настройка положения и размера
            using (var pageGfx = XGraphics.FromPdfPage(page))
            {
                var xRect = new XRect(new XPoint(stampPosition.X, stampPosition.Y), size);
                var stampRect = _stamperMode == PdfStamperMode.Preview ? pageGfx.Transformer.WorldToDefaultPage(xRect) : xRect;

                XForm form = null;
                XGraphics drawGfx;
                PdfRubberStampAnnotation stamp = null;
                if (_stamperMode == PdfStamperMode.Preview)
                {
                    // Создаём XForm нужного размера
                    form = new XForm(_document, new XSize(stampRect.Width, stampRect.Height));
                    drawGfx = XGraphics.FromForm(form);
                    stamp = new PdfRubberStampAnnotation { Rectangle = new PdfRectangle(stampRect) };
                }
                else
                    drawGfx = pageGfx;
                
                using (drawGfx)
                {
                    var paddingLeft = 15;

                    var posRectangle = DrawRoundedRectangle(drawGfx, stampRect);

                    var font = new XFont(Constants.FONT_NAME, 7, XFontStyleEx.Regular);
                    var headerHeight = 28;
                    var currentPos = DrawHeader(drawGfx, font, stampRect, headerHeight, model.StampHeader);
                    currentPos.Y += headerHeight;
                    if (model.Image != null)
                    {
                        // Рисуем логотип
                        using (var imgStream = new MemoryStream())
                        {
                            imgStream.Write(model.Image, 0, model.Image.Length);
                            using (var img = XImage.FromStream(imgStream))
                            {
                                img.Interpolate = false;
                                var imgWidth = img.PixelWidth;
                                var imgHeight = img.PixelHeight;

                                DrawCertificateInfo(drawGfx, font, certInfo, currentPos, stampRect);
                                var logoPaddingTop = 3;
                                drawGfx.DrawImage(img, posRectangle.X + paddingLeft, posRectangle.Y + logoPaddingTop, imgWidth, imgHeight);
                            }
                        }
                    }
                    else
                        DrawCertificateInfo(drawGfx, font, certInfo, currentPos, stampRect);
                }

                if (_stamperMode == PdfStamperMode.Preview)
                {
                    SetAppearance(form, stamp);
                    SetId(stamp, certInfo.Id);
                    // Добавим на страницу
                    page.Annotations.Add(stamp);
                }
            }
        }

        public void AddToProductionStamp(int pageNumber, CertificateInfo certInfo, StampPosition stampPosition, XSize size)
        {
            var page = _document.Pages[pageNumber];

            // настройка положения и размера
            using (var pageGfx = XGraphics.FromPdfPage(page))
            {
                var xRect = new XRect(new XPoint(stampPosition.X, stampPosition.Y), size);
                var stampRect = _stamperMode == PdfStamperMode.Preview ? pageGfx.Transformer.WorldToDefaultPage(xRect) : xRect;

                XForm form = null;
                XGraphics drawGfx;
                PdfRubberStampAnnotation stamp = null;
                if (_stamperMode == PdfStamperMode.Preview)
                {
                    // Создаём XForm нужного размера
                    form = new XForm(_document, new XSize(stampRect.Width, stampRect.Height));
                    drawGfx = XGraphics.FromForm(form);
                    stamp = new PdfRubberStampAnnotation { Rectangle = new PdfRectangle(stampRect) };
                }
                else
                    drawGfx = pageGfx;

                using (drawGfx)
                {
                    var currentPos = DrawRoundedRectangle(drawGfx, stampRect);

                    // Рисуем В ПРОИЗВОДСТВО РАБОТ
                    var font = new XFont(Constants.FONT_NAME, 12, XFontStyleEx.Bold);
                    var padding = 5;
                    currentPos.Y += padding * 4;
                    drawGfx.DrawString(Constants.INTO_PRODUCTION, font, XBrushes.Black, new XPoint(currentPos.X + padding * 7, currentPos.Y));

                    font = new XFont(Constants.FONT_NAME, 7, XFontStyleEx.Regular);
                    currentPos.Y += padding;
                    DrawCertificateInfo(drawGfx, font, certInfo, currentPos, stampRect);
                }

                if (_stamperMode == PdfStamperMode.Preview)
                {
                    SetAppearance(form, stamp);
                    SetId(stamp, certInfo.Id);
                    // Добавим на страницу
                    page.Annotations.Add(stamp);
                }
            }
        }

        private XPoint DrawRoundedRectangle(XGraphics gfx, XRect stampRect)
        {
            var x = _stamperMode == PdfStamperMode.Preview ? 0 : stampRect.X;
            var y = _stamperMode == PdfStamperMode.Preview ? 0 : stampRect.Y;
            // Рисуем рамку штампа 
            gfx.DrawRoundedRectangle(XPens.Black, XBrushes.Transparent, x, y, stampRect.Width, stampRect.Height, 5, 5);
            return new XPoint(x, y);
        }

        private XPoint DrawHeader(XGraphics gfx, XFont font, XRect stampRect, double headerHeight, string header)
        {
            var xStartPos = 62;
            var yStartPos = 3;
            var tf = new XTextFormatter(gfx) { Alignment = XParagraphAlignment.Left };

            var x = _stamperMode == PdfStamperMode.Preview ? xStartPos : stampRect.X + xStartPos;
            var y = _stamperMode == PdfStamperMode.Preview ? yStartPos : stampRect.Y + yStartPos;
            var rect = new XRect(x, y, stampRect.Width - xStartPos, headerHeight);
            // Рисуем текст рядом с логотипом
            tf.DrawString(header, font, XBrushes.Black, rect);
            return new XPoint(x - xStartPos, y);
        }

        private static void DrawCertificateInfo(XGraphics gfx, XFont font, CertificateInfo certInfo, XPoint currentPos, XRect stampRect)
        {
            var paddingTop = 8;
            // Рисуем рамку c заголовком ЭЦП под картинкой
            gfx.DrawRectangle(XPens.Black, XBrushes.Black, currentPos.X, currentPos.Y, stampRect.Width, 10);

            var certTextRect = new XRect(currentPos.X, currentPos.Y, stampRect.Width, 10);
            gfx.DrawString("СВЕДЕНИЯ О СЕРТИФИКАТЕ ЭП", font, XBrushes.White, certTextRect, XStringFormats.TopCenter);

            // Текст после заголовка
            currentPos.Y += certTextRect.Height + paddingTop;
            currentPos.X += 6;
            gfx.DrawString($"Владелец: {certInfo.Subject}", font, XBrushes.Black, new XPoint(currentPos.X, currentPos.Y));
            currentPos.Y += font.Height;
            gfx.DrawString($"Сертификат: {certInfo.Thumbprint}", font, XBrushes.Black, new XPoint(currentPos.X, currentPos.Y));
            currentPos.Y += font.Height;
            gfx.DrawString($"Действителен: с {certInfo.ValidFrom} до {certInfo.ValidTo}", font, XBrushes.Black, new XPoint(currentPos.X, currentPos.Y));
        }

        private static void SetId(PdfDictionary stamp, string stampId)
        {
            if (stamp == null)
                return;

            // Удалим стандартное имя
            stamp.Elements.SetString("/Name", "#" + Guid.NewGuid());
            // Присваиваем уникальный идентификатор через ключ /NM
            stamp.Elements.SetString("/NM", stampId);
        }

        private void SetAppearance(XForm form, PdfDictionary stamp)
        {
            if (form == null)
                return;

            // Устанавливаем appearance‑stream аннотации
            var appearanceStream = new PdfDictionary(_document);
            appearanceStream.Elements.Add("/N", GetPdfFormXObject(form));
            stamp.Elements.Add("/AP", appearanceStream);
        }

        private static PdfFormXObject GetPdfFormXObject(XForm form)
        {
            var type = form.GetType();
            var internalField = type.GetField("_pdfForm", BindingFlags.Instance | BindingFlags.NonPublic);
            var internalValue = internalField?.GetValue(form) as PdfFormXObject;
            return internalValue;
        }

        public void Dispose()
        {
            _document?.Dispose();
            _documentStream?.Dispose();
        }
    }
}