﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using PdfStamper.Models;
using Ascon.Pilot.SDK.Controls;

namespace PdfStamper.ViewModels
{
    public class BaseSettingViewModel : PropertyChangedBase
    {
        private string _xOffsetStr;
        private string _yOffsetStr;
        private string _xOffsetStrPrevious;
        private string _yOffsetStrPrevious;
        private bool _leftBottomCornerButtonChecked;
        private bool _leftTopCornerButtonChecked;
        private bool _rightBottomCornerButtonChecked;
        private bool _rightTopCornerButtonChecked;
        private string _pageNumber;
        private bool _isOnLastPage;
        private bool _putStamp;
        private readonly ISettingListener _settingListener;

        public BaseSettingViewModel(ISettingListener settingListener)
        {
            _settingListener = settingListener;
        }

        public Position SelectedPosition { get; private set; }

        public string XOffsetStr
        {
            get => _xOffsetStr;
            set
            {
                _xOffsetStr = value;
                NotifyOfPropertyChange(nameof(XOffsetStr));
                _settingListener.OnSettingsChanged();
            }
        }

        public string YOffsetStr
        {
            get => _yOffsetStr;
            set
            {
                _yOffsetStr = value;
                NotifyOfPropertyChange(nameof(YOffsetStr));
                _settingListener.OnSettingsChanged();
            }
        }

        public string PageNumber
        {
            get => _pageNumber;
            set
            {
                _pageNumber = value;
                NotifyOfPropertyChange(nameof(PageNumber));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool IsOnLastPage
        {
            get => _isOnLastPage;
            set
            {
                _isOnLastPage = value;
                NotifyOfPropertyChange(nameof(IsOnLastPage));
                _settingListener.OnSettingsChanged();
            }
        }

        public string XOffsetStrPrevious
        {
            get => _xOffsetStrPrevious;
            set
            {
                _xOffsetStrPrevious = value;
                NotifyOfPropertyChange(nameof(XOffsetStrPrevious));
                _settingListener.OnSettingsChanged();
            }
        }

        public string YOffsetStrPrevious
        {
            get => _yOffsetStrPrevious;
            set
            {
                _yOffsetStrPrevious = value;
                NotifyOfPropertyChange(nameof(YOffsetStrPrevious));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool LeftBottomCornerButtonChecked
        {
            get => _leftBottomCornerButtonChecked;
            set
            {
                if (_leftBottomCornerButtonChecked == value)
                    return;

                _leftBottomCornerButtonChecked = value;
                if (value)
                    SelectedPosition = Position.LeftBottom;
                NotifyOfPropertyChange(nameof(LeftBottomCornerButtonChecked));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool LeftTopCornerButtonChecked
        {
            get => _leftTopCornerButtonChecked;
            set
            {
                if (_leftTopCornerButtonChecked == value)
                    return;

                _leftTopCornerButtonChecked = value;
                if (value)
                    SelectedPosition = Position.LeftTop;
                NotifyOfPropertyChange(nameof(LeftTopCornerButtonChecked));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool RightTopCornerButtonChecked
        {
            get => _rightTopCornerButtonChecked;
            set
            {
                if (_rightTopCornerButtonChecked == value)
                    return;

                _rightTopCornerButtonChecked = value;
                if (value)
                    SelectedPosition = Position.RightTop;
                NotifyOfPropertyChange(nameof(RightTopCornerButtonChecked));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool RightBottomCornerButtonChecked
        {
            get => _rightBottomCornerButtonChecked;
            set
            {
                if (_rightBottomCornerButtonChecked == value)
                    return;

                _rightBottomCornerButtonChecked = value;
                if (value)
                    SelectedPosition = Position.RightBottom;
                NotifyOfPropertyChange(nameof(RightBottomCornerButtonChecked));
                _settingListener.OnSettingsChanged();
            }
        }

        public bool PutStamp
        {
            get => _putStamp;
            set
            {
                _putStamp = value;
                NotifyOfPropertyChange(nameof(PutStamp));
                _settingListener.OnSettingsChanged();
            }
        }

        public void SetBaseDefaults(BaseSettingModel defaultModel)
        {
            SelectedPosition = defaultModel.Position;
            _xOffsetStr = defaultModel.OffsetX.ToString();
            _yOffsetStr = defaultModel.OffsetY.ToString();
            _isOnLastPage = defaultModel.IsOnLastPage;
            _pageNumber = defaultModel.PageNumber.ToString();
            _xOffsetStrPrevious = defaultModel.OffsetPreviousX.ToString();
            _yOffsetStrPrevious = defaultModel.OffsetPreviousY.ToString();
            _leftTopCornerButtonChecked = defaultModel.Position == Position.LeftTop;
            _leftBottomCornerButtonChecked = defaultModel.Position == Position.LeftBottom;
            _rightTopCornerButtonChecked = defaultModel.Position == Position.RightTop;
            _rightBottomCornerButtonChecked = defaultModel.Position == Position.RightBottom;
            _putStamp = defaultModel.PutStamp;
        }

        public void FillBaseWithData(BaseSettingModel model)
        {
            _xOffsetStr = model.OffsetX.ToString();
            _yOffsetStr = model.OffsetY.ToString();
            _isOnLastPage = model.IsOnLastPage;
            _pageNumber = model.PageNumber.ToString();
            _xOffsetStrPrevious = model.OffsetPreviousX.ToString();
            _yOffsetStrPrevious = model.OffsetPreviousY.ToString();
            SelectedPosition = model.Position;
            _putStamp = model.PutStamp;

            switch (model.Position)
            {
                case Position.LeftBottom:
                    _leftBottomCornerButtonChecked = true;
                    break;
                case Position.LeftTop:
                    _leftTopCornerButtonChecked = true;
                    break;
                case Position.RightBottom:
                    _rightBottomCornerButtonChecked = true;
                    break;
                case Position.RightTop:
                    _rightTopCornerButtonChecked = true;
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }
    }
}