/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using Ascon.Pilot.SDK;
using DynamicData;
using Moq;
using NUnit.Framework;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.Data;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.ViewModels;

namespace ProjectCloneWizard.Tests
{
    [TestFixture]
    class ProjectCompositionPageViewModelTests
    {
        private Mock<IDataObjectService> _dataObjectServiceMock;
        private Mock<IDataObjectServiceFactory> _dataObjectServiceFactoryMock;

        [SetUp]
        public void SetUp()
        {
            _dataObjectServiceMock = new Mock<IDataObjectService>();
            var dataObjects = new SourceCache<IDataObject, Guid>(x => x.Id);
            _dataObjectServiceMock.Setup(s => s.DataObjects).Returns(dataObjects.AsObservableCache);

            _dataObjectServiceFactoryMock = new Mock<IDataObjectServiceFactory>();
            _dataObjectServiceFactoryMock.Setup(f => f.GetService()).Returns(_dataObjectServiceMock.Object);
        }

        [Test]
        public void should_return_correct_values()
        {
            var creationInfo = new CreationInfo
            {
                Template = TestTools.GetNodeViewModel().Source
            };

            var viewModel = new ProjectCompositionPageViewModel(creationInfo, _dataObjectServiceFactoryMock.Object);

            var expected = "Шаг 2 из 4. Выберите необходимые элементы состава";
            var actual = viewModel.GetTitle();
            Assert.AreEqual(expected, actual);

            actual = viewModel.GetTemplateName();
            Assert.IsNull(actual);
        }

        /*
        [Test]
        public void should_set_copy_access_to_result()
        {
            var creationInfo = new CreationInfo
            {
                Template = TestTools.GetNodeViewModel()
            };

            var viewModel = new ProjectCompositionPageViewModel(creationInfo, _dataObjectServiceFactoryMock.Object);

            viewModel.CopyAccessRights = true;
            Assert.IsTrue(creationInfo.CopyAccessForObjects);

            viewModel.CopyAccessRights = false;
            Assert.IsFalse(creationInfo.CopyAccessForObjects);
        }
        */

        [Test]
        public void should_not_load_tree_twice()
        {
            var creationInfo = new CreationInfo
            {
                Template = TestTools.GetNodeViewModel(Guid.NewGuid()).Source
            };

            var viewModel = new ProjectCompositionPageViewModel(creationInfo, _dataObjectServiceFactoryMock.Object);
            viewModel.Load();

            var first = viewModel.ObjectsTreeViewModel;
            Assert.NotNull(first);

            viewModel.Load();
            Assert.AreSame(first, viewModel.ObjectsTreeViewModel);
        }

        [Test, Ignore("Popov Мир пока не готов. Надо переписать тест.")]
        public void should_set_object_nodes_to_result()
        {
            var creationInfo = new CreationInfo
            {
                Template = TestTools.GetNodeViewModel(Guid.NewGuid()).Source
            };

            var viewModel = new ProjectCompositionPageViewModel(creationInfo, _dataObjectServiceFactoryMock.Object);
            viewModel.Load();

            Assert.NotNull(creationInfo.ObjectsNodes);
            Assert.AreEqual(creationInfo.ObjectsNodes.Count, viewModel.ObjectsTreeViewModel.Nodes.Count);
        }
    }
}
