/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Reactive.Subjects;
using System.Windows.Input;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.Commands;
using DynamicData;
using DynamicData.Binding;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.Data;

namespace ProjectCloneWizard.ObjectsTree
{
    public class ObjectsTreeViewModel : AbstractNotifyPropertyChanged, IDisposable
    {
        private readonly ReadOnlyObservableCollection<NodeViewModel> _nodes;
        private readonly IDataObjectService _dataObjectService;
        private readonly CompositeDisposable _compositeDisposable = new CompositeDisposable();
        private readonly Subject<List<NodeViewModel>> _checkedListSubject = new Subject<List<NodeViewModel>>();
        private readonly Dictionary<Guid, NodeViewModel> _checkedNodes = new Dictionary<Guid, NodeViewModel>();

        private readonly DelegateCommand _collapseAllCommand;
        private readonly DelegateCommand _expandAllCommand;
        private readonly DelegateCommand _checkAllCommand;
        private readonly DelegateCommand _uncheckAllCommand;
        private readonly DelegateCommand _checkOnlyFoldersCommand;
        
        private NodeViewModel _selectedItem = null;
        private bool _isLoading;

        public event EventHandler SelectedItemChanged;

        public ObjectsTreeViewModel(IDataObjectService dataObjectService, bool isExpand)
        {
            _collapseAllCommand = new DelegateCommand(CollapseAll);
            _expandAllCommand = new DelegateCommand(ExpandAll);
            _checkAllCommand = new DelegateCommand(CheckAll);
            _uncheckAllCommand = new DelegateCommand(UncheckAll);
            _checkOnlyFoldersCommand = new DelegateCommand(CheckOnlyFolders);
            ToolbarItemsSource = new ObservableCollection<object>();

            _dataObjectService = dataObjectService;
            _compositeDisposable.Add(_dataObjectService);

            var disposable = _dataObjectService.DataObjects.Connect()
                .TransformToTree(o => o.ParentId)
                .Transform(node => CreateNodeViewModel(node, _dataObjectService, isExpand))
                .Sort(new NodeViewModelComparer())
                .Bind(out _nodes)
                .DisposeMany()
                .Subscribe(set => { IsLoading = false; }, () => { IsLoading = false; });

            _compositeDisposable.Add(disposable);
        }

        public ObservableCollection<object> ToolbarItemsSource { get; }

        public ReadOnlyObservableCollection<NodeViewModel> Nodes => _nodes;

        public IObservable<List<NodeViewModel>> CheckedList => _checkedListSubject;

        public ICommand CollapseAllCommand => _collapseAllCommand;

        public ICommand ExpandAllCommand => _expandAllCommand;

        public ICommand CheckAllCommand => _checkAllCommand;

        public ICommand UncheckAllCommand => _uncheckAllCommand;

        public ICommand CheckOnlyFoldersCommand => _checkOnlyFoldersCommand;

        public NodeViewModel SelectedItem
        {
            get { return _selectedItem; }
            set
            {
                _selectedItem = value;
                SelectedItemChanged?.Invoke(this, EventArgs.Empty);
            }
        }

        public bool IsLoading
        {
            get { return _isLoading; }
            set
            {
                _isLoading = value;
                CheckAll();
                OnPropertyChanged();
            }
        }

        public void Load(Guid id)
        {
            if (_nodes.Any())
                return;

            IsLoading = true;
            _dataObjectService.GetElementStructureAsync(id, () => IsLoading = false);
        }

        public void LoadItem(Guid id, bool isAddParent)
        {
            IsLoading = true;
            _dataObjectService.GetElementStructureAsync(id, () => IsLoading = false, isAddParent);
        }

        public void Dispose()
        {
            _compositeDisposable.Dispose();
        }

        private void CollapseAll()
        {
            foreach (var node in Nodes)
            {
                SetIsExpanded(node, false);
            }
        }

        private void ExpandAll()
        {
            foreach (var node in Nodes)
            {
                SetIsExpanded(node, true);
            }
        }

        private void SetIsExpanded(NodeViewModel parent, bool isExpanded)
        {
            if (parent.Children == null)
                return;

            foreach (var child in parent.Children)
            {
                child.IsExpanded = isExpanded;
                SetIsExpanded(child, isExpanded);
            }

            parent.IsExpanded = isExpanded;
        }

        private void UncheckAll()
        {
            foreach (var node in Nodes)
            {
                SetIsChecked(node, false, false);
            }
        }

        private void CheckAll()
        {
            foreach (var node in Nodes)
            {
                SetIsChecked(node, true, false);
            }
        }

        private void CheckOnlyFolders()
        {
            UncheckAll();
            foreach (var node in Nodes)
            {
                SetIsChecked(node, true, true);
            }
        }

        private void SetIsChecked(NodeViewModel parent, bool isChecked, bool foldersOnly)
        {
            if (parent.Children == null)
                return;

            foreach (var child in parent.Children)
            {
                if (foldersOnly && !child.Type.IsFolder())
                    continue;

                child.IsChecked = isChecked;
                SetIsChecked(child, isChecked, foldersOnly);
            }

            parent.IsChecked = isChecked;
        }

        private NodeViewModel CreateNodeViewModel(Node<IDataObject, Guid> node, IDataObjectService dataObjectService, bool isExpand, NodeViewModel parent = null)
        {
            var vm =  new NodeViewModel(node, dataObjectService, isExpand, parent);
            var subscription = vm.IsCheckedObservable.Subscribe((value) =>
            {
                if (value)
                    _checkedNodes[vm.Id] = vm;
                else
                    _checkedNodes.Remove(vm.Id);

                _checkedListSubject.OnNext(_checkedNodes.Values.ToList());
            });
            _compositeDisposable.Add(subscription);
            return vm;
        }
    }
}
