/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Menu;
using DynamicData;
using ProjectCloneWizard.Common;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace ProjectCloneWizard
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class ProjectCloneMenu : IMenu<ObjectsViewContext>
    {
        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly ICopyStructureService _copyStructureService;
        private readonly IIconsProvider _iconsProvider;
        private readonly IProjectCloneCreator _projectCloneCreator;

        [ImportingConstructor]
        public ProjectCloneMenu(ITabServiceProvider tabServiceProvider, IProjectCloneCreator projectCloneCreator, 
            ICopyStructureService copyStructureService, IIconsProvider iconsProvider)
        {
            _tabServiceProvider = tabServiceProvider;
            _copyStructureService = copyStructureService;
            _iconsProvider = iconsProvider;
            _projectCloneCreator = projectCloneCreator;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();
            if (objects.Count != 1)
                return;

            AddMenuCreate(builder, objects);
            AddMenuCopy(builder);
            AddMenuPaste(builder);
        }

        public async void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name == Const.COMMAND_UPDATE_NAME || name == Const.COMMAND_NAME)
            {
                var selected = context.SelectedObjects.First();
                _projectCloneCreator.ShowWizardDialog(selected);
            }

            if (name == Const.COMMAND_COPY_STRUCTURE_HERE_NAME)
            {
                var selected = context.SelectedObjects.First();
                var idToShow = await _copyStructureService.CopyStructure(selected);
                _tabServiceProvider.ShowElement(idToShow);
            }

            if (name == Const.COMMAND_COPY_STRUCTURE_NAME)
            {
                var selected = context.SelectedObjects.FirstOrDefault();
                _copyStructureService.SetElementToCopy(selected);
            }

            if (name == Const.COMMAND_PASTE_STRUCTURE_NAME)
            {
                var copied = _copyStructureService.GetElementToCopy();
                var selected = context.SelectedObjects.First();
                var idToShow = await _copyStructureService.CopyStructure(copied, selected);
                _tabServiceProvider.ShowElement(idToShow);
            }
        }

        private void AddMenuCreate(IMenuBuilder builder, List<IDataObject> objects)
        {
            var menu = builder.ItemNames.ToList();
            if (!menu.Contains("miCreate"))
                return;

            var subMenu = builder.GetItem("miCreate");
            var index = subMenu.Count;

            var smartFolderIndex = subMenu.ItemNames.ToList().IndexOf("miCreateSmartfoldertype");
            if (smartFolderIndex != -1)
                index = smartFolderIndex == 0 ? subMenu.Count - 1 : smartFolderIndex - 1;

            var selected = objects.First();
            if (selected.Type.IsProject)
            {
                subMenu.AddItem(Const.COMMAND_UPDATE_NAME, index)
                    .WithHeader(Const.COMMAND_UPDATE_TITLE)
                    .WithIcon(IconLoader.GetIcon(_projectCloneCreator.Icon));
            }
            else
            {
                subMenu.AddItem(Const.COMMAND_NAME, index)
                    .WithHeader(Const.COMMAND_TITLE)
                    .WithIcon(IconLoader.GetIcon(_projectCloneCreator.Icon));
            }
        }

        private void AddMenuCopy(IMenuBuilder builder)
        {
            var index = builder.ItemNames.IndexOf("miCopy");
            if (index == -1)
                return;

            builder.AddItem(Const.COMMAND_COPY_STRUCTURE_HERE_NAME, ++index)
                .WithHeader(Const.COMMAND_COPY_STRUCTURE_HERE_TITLE)
                .WithIcon(_iconsProvider.GetCopyStructureHereIcon());

            builder.AddItem(Const.COMMAND_COPY_STRUCTURE_NAME, ++index)
                .WithHeader(Const.COMMAND_COPY_STRUCTURE_TITLE)
                .WithIcon(_iconsProvider.GetCopyStructureIcon());
        }

        private void AddMenuPaste(IMenuBuilder builder)
        {
            var index = builder.ItemNames.IndexOf("miPasteAsShortcut");
            if (index == -1)
                return;

            builder.AddItem(Const.COMMAND_PASTE_STRUCTURE_NAME, ++index)
                .WithHeader(Const.COMMAND_PASTE_STRUCTURE_TITLE)
                .WithIcon(_iconsProvider.GetPasteStructureIcon())
                .WithIsEnabled(_copyStructureService.GetElementToCopy() != null);
        }
    }
}
