/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using Ascon.Pilot.SDK.Controls.Commands;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.Wizard;

namespace ProjectCloneWizard.ViewModels
{
    class FilesAndFolderPageViewModel : PropertyChangedBase, IWizardCaptionProvider, IDisposable
    {
        private readonly CreationInfo _creationInfo;
        private readonly IDataObjectServiceFactory _serviceFactory;
        private ObjectsTreeViewModel _objectsTreeViewModel;
        private Guid _currentTemplateId;
        private ToolbarToggleButtonItem _copyToolbarItem;
        private readonly DelegateCommand _copyAccessRightsCommand;
        private IDisposable _subscription;

        public FilesAndFolderPageViewModel(CreationInfo creationInfo, IDataObjectServiceFactory serviceFactory)
        {
            _creationInfo = creationInfo;
            _serviceFactory = serviceFactory;
            _copyAccessRightsCommand = new DelegateCommand(CopyAccess);
        }

        public ObjectsTreeViewModel ObjectsTreeViewModel
        {
            get { return _objectsTreeViewModel; }
            set
            {
                _objectsTreeViewModel = value;
                NotifyOfPropertyChanged(() => ObjectsTreeViewModel);
            }
        }

        public void Load()
        {
            if (_currentTemplateId == _creationInfo.Template.Id)
                return;

            _currentTemplateId = _creationInfo.Template.Id;

            ObjectsTreeViewModel = new ObjectsTreeViewModel(_serviceFactory.GetService(), true);
            _subscription = ObjectsTreeViewModel.CheckedList.Subscribe(list => { _creationInfo.StorageNodes = list; });
            ObjectsTreeViewModel.Load(_currentTemplateId);

            if (_creationInfo.ObjectsNodes != null)
            {
                var ecmDocs = _creationInfo.ObjectsNodes.Where(n => n.Type.IsEcmDocument());
                foreach (var node in ecmDocs)
                {
                    ObjectsTreeViewModel.LoadItem(node.Id, true);
                }
            }

            FillToolbar();
        }


        public string GetTitle()
        {
            return "Шаг 3 из 4. Выберите необходимые папки и файлы";
        }

        public string GetTemplateName()
        {
            return _creationInfo.Template.DisplayName;
        }

        private void FillToolbar()
        {
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("CollapseAll")
            {
                Header = "Свернуть все",
                Icon = Icons.Instance.CollapseAllIcon,
                Command = _objectsTreeViewModel.CollapseAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("ExpandAll")
            {
                Header = "Развернуть все",
                Icon = Icons.Instance.ExpandAllIcon,
                Command = _objectsTreeViewModel.ExpandAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("CheckAll")
            {
                Header = "Выбрать все",
                Icon = Icons.Instance.CheckAllIcon,
                Command = _objectsTreeViewModel.CheckAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("UncheckAll")
            {
                Header = "Снять все",
                Icon = Icons.Instance.UncheckAllIcon,
                Command = _objectsTreeViewModel.UncheckAllCommand,
            });
            _objectsTreeViewModel.ToolbarItemsSource.Add(new ToolbarButtonItem("CheckFoldersOnly")
            {
                Header = "Только папки",
                Icon = Icons.Instance.CheckFoldersIcon,
                Command = _objectsTreeViewModel.CheckOnlyFoldersCommand
            });
            _copyToolbarItem = new ToolbarToggleButtonItem("CopyAccess")
            {
                Header = "Скопировать права доступа для выбранных папок и файлов",
                Icon = Icons.Instance.UncheckAllIcon,
                IsChecked = false,
                Command = _copyAccessRightsCommand
            };
            _objectsTreeViewModel.ToolbarItemsSource.Add(_copyToolbarItem);
        }

        private void CopyAccess()
        {
            _copyToolbarItem.IsChecked = !_copyToolbarItem.IsChecked;
            _copyToolbarItem.Icon = _copyToolbarItem.IsChecked ? Icons.Instance.CheckAllIcon : Icons.Instance.UncheckAllIcon;
            _creationInfo.CopyAccessForFiles = _copyToolbarItem.IsChecked;
        }

        public void Dispose()
        {
            _objectsTreeViewModel?.Dispose();
            _subscription?.Dispose();
        }
    }
}
