/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using Ascon.Pilot.Theme.Controls.Windows.Tools;
using Ascon.Pilot.Theme.Tools;
using ProjectCloneWizard.Wizard;

namespace ProjectCloneWizard
{
    /// <summary>
    /// Interaction logic for WizardControl.xaml
    /// </summary>
    public partial class WizardControl : UserControl
    {
        private readonly Stack<Page> _pages = new Stack<Page>();
        private readonly List<Page> _pagesList = new List<Page>();
        private readonly string _nextButtonDefaultCaption;
        private IWizardNavigationService _navService;

        public WizardControl()
        {
            InitializeComponent();
            Loaded += OnLoaded;
            Frame.Navigated += NavigationServiceOnNavigated;
            _nextButtonDefaultCaption = NextButton.Content.ToString();
        }

        private void NavigationServiceOnNavigated(object sender, NavigationEventArgs e)
        {
            _navService = GetNavigationService();
            BackButton.IsEnabled = Frame.CanGoBack;

            if (_navService != null)
                BackButton.IsEnabled = _navService.CanGoBack();

            UpdateTitle();
            UpdateNextButton();
            UpdateDefaultButton();

            if (_navService != null)
            {
                _navService.UpdateState += NavServiceOnUpdateState;
                _navService.NavigateNext += NavServiceOnNavigateNext;
            }

            if (Frame.BackStack == null)
                return;

            if (_pages.Count == 0)
                NextButton.IsEnabled = Frame.CanGoForward;

            if (_pages.Count != 0)
                return;

            CloseButton.Content = "Готово";
            BackButton.Visibility = Visibility.Collapsed;
            NextButton.Visibility = Visibility.Collapsed;
            UpdateDefaultButton();
        }

        private void UpdateDefaultButton()
        {
            NextButton.IsDefault = true;
            CloseButton.IsDefault = NextButton.IsEnabled == false || NextButton.IsVisible == false;
        }

        private void UpdateNextButton()
        {
            NextButton.Content = _nextButtonDefaultCaption;
            var nextCaption = _navService?.NextButtonCaption();
            if (!string.IsNullOrEmpty(nextCaption))
                NextButton.Content = nextCaption;
        }

        private void NavServiceOnUpdateState(object sender, EventArgs eventArgs)
        {
           if (_navService != null)
                NextButton.IsEnabled = _navService.CanGoNext();

           UpdateDefaultButton();
        }

        private void NavServiceOnNavigateNext(object sender, EventArgs eventArgs)
        {
            GoForward();
        }

        private void UpdateTitle()
        {
            TitleTextBlock.Text = null;
            TemplateTextBlock.Text = null;

            var provider = GetCaptionProvider();
            if (provider == null)
                return;

            TitleTextBlock.Text = provider.GetTitle();
            TemplateTextBlock.Text = provider.GetTemplateName();
        }

        private void OnLoaded(object sender, RoutedEventArgs routedEventArgs)
        {
            _pagesList.Reverse();
            foreach (var page in _pagesList)
            {
                _pages.Push(page);
            }

            if (_pages.Any())
                Frame.Content = _pages.Pop();
        }

        public List<Page> Pages
        {
            get { return _pagesList; }

        }
        private void Back_OnClick(object sender, RoutedEventArgs e)
        {
            NextButton.IsEnabled = true;

            if (Frame.CanGoBack)
                Frame.GoBack();
        }

        private void NextButton_OnClick(object sender, RoutedEventArgs e)
        {
            GoForward();
        }

        private void GoForward()
        {
            if (!CanGoNext())
                return;

            if (Frame.CanGoForward)
            {
                Frame.GoForward();
                return;
            }

            if (_pages.Count == 0)
                return;

            var page = _pages.Pop();
            Frame.Navigate(page);
        }

        private bool CanGoNext()
        {
            return _navService == null || _navService.CanGoNext();
        }

        private IWizardNavigationService GetNavigationService()
        {
            var page = (Page)Frame.Content;
            var navService = page as IWizardNavigationService ?? page.DataContext as IWizardNavigationService;
            return navService;
        }

        private IWizardCaptionProvider GetCaptionProvider()
        {
            var page = (Page)Frame.Content;
            var provider = page as IWizardCaptionProvider ?? page.DataContext as IWizardCaptionProvider;
            return provider;
        }

        private void CancelButton_OnClick(object sender, RoutedEventArgs e)
        {
            var nav = GetNavigationService();
            nav?.Cancel();

            var result = (string)CloseButton.Content == "Готово";
            var window = LayoutHelper.FindParent<Window>(this);
            window.CloseWithResult(result);
        }
    }
}
