/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using DynamicData;
using DynamicData.Binding;
using DynamicData.Kernel;

namespace ShowObjectsTree.DynamicData
{
    public class NodeViewModel : AbstractNotifyPropertyChanged, IDisposable
    {
        private readonly Node<DataObject, Guid> _node;
        private ReadOnlyObservableCollection<NodeViewModel> _children;
        private bool _isExpanded;
        private readonly IDisposable _cleanUp;

        public NodeViewModel(Node<DataObject, Guid> node, IDataObjectService dataObjectService, NodeViewModel parent = null)
        {
            _node = node;
            Parent = parent;

            var childrenLoader = new Lazy<IDisposable>(() => node.Children.Connect()
                    .Transform(x => new NodeViewModel(x, dataObjectService, this))
                    .Sort(new NodeViewModelComparer())
                    .Bind(out _children)
                    .DisposeMany()
                    .Subscribe());

            var shouldExpand = node.IsRoot
                ? Observable.Return(true)
                : Parent.Value.WhenValueChanged(x => x.IsExpanded);
            
            var expander = shouldExpand
                    .Where(isExpanded => isExpanded)
                    .Take(1)
                    .Subscribe(x =>
                    {
                        dataObjectService.Load(node.Item.Children);
                        //force lazy loading
                        var value = childrenLoader.Value;
                    });

            _cleanUp = Disposable.Create(() =>
            {
                expander.Dispose();
                if (childrenLoader.IsValueCreated)
                    childrenLoader.Value.Dispose();
            });
        }

        public string Title => _node.Item.Title;
        public byte[] Icon => _node.Item.Icon;
        public int Sort => _node.Item.Sort;
        public Optional<NodeViewModel> Parent { get; }
        public ReadOnlyObservableCollection<NodeViewModel> Children => _children;
        
        public bool IsExpanded
        {
            get { return _isExpanded; }
            set { SetAndRaise(ref _isExpanded, value); }
        }

        public void Dispose()
        {
            _cleanUp.Dispose();
        }
    }
}
