/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Automation;

namespace SignMeAsActivity
{
    [Export(typeof(IAutomationActivity))]
    public class SignMeAsActivity : AutomationActivity
    {
        private readonly IDigitalSigner _digitalSigner;
        private readonly IObjectsRepository _repository;
        private readonly IFileProvider _fileProvider;

        [ImportingConstructor]
        public SignMeAsActivity(IDigitalSigner digitalSigner, IObjectsRepository repository, IFileProvider fileProvider)
        {
            _digitalSigner = digitalSigner;
            _repository = repository;
            _fileProvider = fileProvider;
        }

        public override string Name => nameof(SignMeAsActivity);

        public override Task RunAsync(IObjectModifier modifier, IAutomationBackend backend, IAutomationEventContext context, TriggerType triggerType)
        {
            var sourceObject = backend.GetObject(context.Source.Id);
            var person = context.InitiatingPerson;
            // check that my task
            if (sourceObject.Attributes.TryGetValue(SystemTaskAttributes.EXECUTOR_POSITION, out var executorAttribute))
            {
                if (executorAttribute is int[] executors)
                {
                    var executor = executors.FirstOrDefault(); 
                    if (person.Positions.FirstOrDefault(p => p.Position == executor) == null)
                        return Task.CompletedTask;
                }
            }

            var attachments = ActivityExtensions.GetObjectsToExecute(context, backend, this).ToList();
            if (!attachments.Any())
            {
                throw new AutomationCanceledException();
            }

            var attachmentsToSign = attachments.Where(a => a.ActualFileSnapshot.Files.Any(f => f.SignatureRequests.Any(s => s.PositionId == context.InitiatingPerson.MainPosition.Position)));
            var positions = _repository.GetCurrentPerson().Positions.Select(p => p.Position).ToList();
            var cert = _digitalSigner.GetAutomationCertificate();
            if (cert == null)
                return Task.CompletedTask;

            foreach (var doc in attachmentsToSign)
            {
                var file = doc.ActualFileSnapshot.Files.FirstOrDefault(x => x.SignatureRequests.Any(sr => !sr.Signs.Any()));
                if (file == null)
                    continue;

                var signatureRequests = file.SignatureRequests.Where(x => positions.Contains(x.PositionId) && !x.Signs.Any()).ToList();
                if (!signatureRequests.Any())
                    continue;


                using (var stream = _fileProvider.OpenRead(file))
                {
                    _digitalSigner.SignByAutomation(modifier, doc.Id, file, stream, cert, signatureRequests);
                }
            }

            return Task.CompletedTask;
        }
    }
}
