/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api.Contracts;

namespace Pilot.Xps.Domain.Context
{
    public interface IFileLoader
    {
        byte[] Download(INFile inFile);
        byte[] Download(Guid fileId, long size);
    }

    public class FileLoader : IFileLoader
    {
        private static int CHUNK_SIZE = 1024 * 1024; // 1 Mb

        private readonly IFileArchiveApi _fileArchiveApi;

        public FileLoader(IFileArchiveApi fileArchiveApi)
        {
            _fileArchiveApi = fileArchiveApi;
        }

        public byte[] Download(INFile inFile)
        {
            return Download(inFile.Id, inFile.Size);
        }

        public byte[] Download(Guid fileId, long size)
        {
            using (var stream = new MemoryStream())
            using (BinaryWriter file = new BinaryWriter(stream))
            {
                var filePos = _fileArchiveApi.GetFilePosition(fileId);
                long fileSize = size;
                while (fileSize > 0)
                {
                    int chunkSize = fileSize > CHUNK_SIZE ? CHUNK_SIZE : (int)fileSize;
                    var data = _fileArchiveApi.GetFileChunk(fileId, filePos + size - fileSize, chunkSize);
                    file.Write(data);
                    fileSize -= chunkSize;
                }

                return stream.ToArray();
            }
        }

        private static string GetTempPath(string fileName)
        {
            return Path.Combine(DirectoryProvider.GetTempDirectory(), fileName);
        }
    }
}
