/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Concurrent;
using System.IO;

namespace Pilot.Xps.Domain
{
    public class DocumentContext : IDisposable
    {
        private readonly ConcurrentDictionary<DocumentUsageArea, Stream> _streams = new ConcurrentDictionary<DocumentUsageArea, Stream>();
        internal readonly ConcurrentBag<Stream> _abandonedStreams = new ConcurrentBag<Stream>();

        public DocumentContext(Stream originalStream)
        {
            _streams.TryAdd(DocumentUsageArea.Original, originalStream);
        }

        public RegistrableStream CreateStream(DocumentUsageArea area)
        {
            var stream = new RegistrableMemoryStream();
            if (!_streams.TryAdd(area, stream))
                throw new InvalidOperationException("Stream with the same name has already been added.");
            return new PositionKeeperStream(stream);
        }

        public Stream GetStream(DocumentUsageArea usageArea)
        {
            if (_streams.TryGetValue(usageArea, out var outStream))
            {
                if (outStream is RegistrableMemoryStream registrableStream && !registrableStream.IsRegistered)
                    return null;
                return new PositionKeeperStream(outStream);
            }
            return null;
        }

        public Stream CloneStream(DocumentUsageArea source, DocumentUsageArea destination)
        {
            if (_streams.TryGetValue(source, out var sourceStream))
            {
                if (_streams.TryAdd(destination, sourceStream))
                    return GetStream(destination);
                throw new InvalidOperationException("Destination stream already exists");
            }
            throw new InvalidOperationException("Source stream doesn't exist");
        }

        public bool StreamExists(DocumentUsageArea usageArea)
        {
            return _streams.ContainsKey(usageArea);
        }

        public void Dispose()
        {
            foreach (var stream in _streams)
            {
                stream.Value.Dispose();
            }
            _streams.Clear();

            while (_abandonedStreams.TryTake(out var abandonedStream))
            {
                abandonedStream.Dispose();
            }
        }

        public RegistrableStream ModifyStream(DocumentUsageArea usageArea)
        {
            if (_streams.TryGetValue(usageArea, out var outStream))
            {
                lock (outStream)
                {
                    var copy = new RegistrableMemoryStream();
                    var position = outStream.Position;
                    outStream.Position = 0;
                    outStream.CopyTo(copy);
                    _abandonedStreams.Add(_streams[usageArea]);
                    _streams[usageArea] = copy;
                    outStream.Position = position;
                    copy.Position = 0;
                    return new PositionKeeperStream(copy);
                }
            }
            throw new InvalidOperationException("There is no such stream.");
        }

        public void RemoveStream(DocumentUsageArea area)
        {
            if (_streams.TryRemove(area, out var outStream))
            {
                outStream.Dispose();
            }
        }
    }

    public enum DocumentUsageArea
    {
        Original,
        TextOnly,
        ImagesOnly,
        GeometryOnly,
        TextAndGeometry,
        Signatures,
        Annotaitons,
        Empty
    }

    public class PositionKeeperStream : RegistrableStream
    {
        private readonly Stream _stream;

        public PositionKeeperStream(Stream stream)
        {
            _stream = stream;
        }

        public override void Flush()
        {
            lock (_stream)
            {
                _stream.Flush();
            }
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            lock (_stream)
            {
                _stream.Position = Position;
                var seekResult = _stream.Seek(offset, origin);
                Position = _stream.Position;
                return seekResult;
            }
        }

        public override void SetLength(long value)
        {
            lock (_stream)
            {
                _stream.SetLength(value);
            }
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            lock (_stream)
            {
                _stream.Position = Position;
                var read = _stream.Read(buffer, offset, count);
                Position = _stream.Position;
                return read;
            }
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            lock (_stream)
            {
                _stream.Position = Position;
                _stream.Write(buffer, offset, count);
                Position = _stream.Position;
            }
        }

        public override bool CanRead
        {
            get
            {
                lock (_stream)
                {
                    return _stream.CanRead;
                }
            }
        }

        public override bool CanSeek
        {
            get
            {
                lock (_stream)
                {
                    return _stream.CanSeek;
                }
            }
        }

        public override bool CanWrite
        {
            get
            {
                lock (_stream)
                {
                    return _stream.CanWrite;
                }
            }
        }

        public override long Length
        {
            get
            {
                lock (_stream)
                {
                    return _stream.Length;
                }
            }
        }

        public override long Position { get; set; }

        public override bool IsRegistered
        {
            get
            {
                return _stream is RegistrableStream registrableStream && registrableStream.IsRegistered;
            }
        }

        public override void Register()
        {
            if (_stream is RegistrableStream registrableStream)
                registrableStream.Register();
            else
                throw new NotSupportedException();
        }
    }

    public class RegistrableMemoryStream : RegistrableStream
    {
        private readonly MemoryStream _stream;
        private bool _isReadOnly;
        private bool _isRegistered;

        public RegistrableMemoryStream()
        {
            _stream = new MemoryStream();
        }

        public override bool IsRegistered => _isRegistered;

        public override void Register()
        {
            _isReadOnly = true;
            _isRegistered = true;
        }

        public override void Flush()
        {
            _stream.Flush();
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            return _stream.Seek(offset, origin);
        }

        public override void SetLength(long value)
        {
            DemandWrite();
            _stream.SetLength(value);
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            return _stream.Read(buffer, offset, count);
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            DemandWrite();
            _stream.Write(buffer, offset, count);
        }

        public override bool CanRead => _stream.CanRead;
        public override bool CanSeek => _stream.CanSeek;
        public override bool CanWrite => !_isReadOnly;
        public override long Length => _stream.Length;
        public override long Position 
        { 
            get => _stream.Position;
            set => _stream.Position = value;
        }
        private void DemandWrite()
        {
            if (_isReadOnly)
                throw new InvalidOperationException("Stream is already registered. Write operation is not allowed.");
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            _stream.Dispose();
        }
    }

    public abstract class RegistrableStream : Stream
    {
        public abstract bool IsRegistered { get; }
        public abstract void Register();
    }
}
