/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Collections.Generic;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Xml.Serialization;

namespace Pilot.Xps.Domain.GraphicLayer
{
    class FloatingElementsReaderWriter
    {
        public void WriteLabel(Stream outStream, string label, Dictionary<string, object> attributes)
        {
            WriteFloatingElement(outStream, label, attributes, XpsConstants.TEXT_LABELS_RELS_TYPE, XpsConstants.TEXT_LABELS_DATA_FILE, XpsConstants.TEXT_LABELS_CONTENT_TYPE);
        }

        public void WriteBarcode(Stream outStream, string barcode, Dictionary<string, object> attributes)
        {
            WriteFloatingElement(outStream, barcode, attributes, XpsConstants.BARCODE_RELS_TYPE, XpsConstants.BARCODE_DATA_FILE, XpsConstants.BARCODE_CONTENT_TYPE);
        }
        public string ReadLabel(Stream stream)
        {
            return ReadFloatingElement(stream, XpsConstants.TEXT_LABELS_RELS_TYPE, XpsConstants.TEXT_LABELS_DATA_FILE, XpsConstants.TEXT_LABELS_CONTENT_TYPE);
        }
        public string ReadBarcode(Stream stream)
        {
            return ReadFloatingElement(stream, XpsConstants.BARCODE_RELS_TYPE, XpsConstants.BARCODE_DATA_FILE, XpsConstants.BARCODE_CONTENT_TYPE);
        }
        public void WriteGraphicLayer(Stream outStream, IList<GraphicLayerElement> floatingGraphicElements)
        {
            using (var package = Package.Open(outStream, FileMode.Open, FileAccess.ReadWrite))
            {
                var strategy = new GraphicLayerXpsPackagingStrategy(package);
                foreach (var element in floatingGraphicElements.ToList())
                {
                    using (var stream = strategy.GetPackageStream(element.ContentType, element.ContentId, true))
                        element.GraphicLayerElementContent.GetStream().CopyTo(stream);
                }

                using (var stream = XpsTools.GetStreamInPackage(package, XpsConstants.GRAPHIC_LAYER_ELEMENT_RELS_TYPE,
                        XpsConstants.GRAPHIC_LAYER_ELEMENT_DATA_FILE, XpsConstants.GRAPHIC_LAYER_ELEMENT_CONTENT_TYPE, true))
                {
                    var serializer = new XmlSerializer(typeof(GraphicLayerElementSaveObj));
                    serializer.Serialize(stream, new GraphicLayerElementSaveObj
                    {
                        GraphicLayerElements = new List<GraphicLayerElement>(floatingGraphicElements)
                    });
                }
            }
        }
        private static void WriteFloatingElement(Stream outStream, string floatingElement, Dictionary<string, object> attributes, string relsType, string dataFile, string contentType)
        {
            if (!string.IsNullOrEmpty(floatingElement))
            {
                using (var package = Package.Open(outStream, FileMode.Open, FileAccess.ReadWrite))
                {
                    var floatingLabelsStream = XpsTools.GetStreamInPackage(package, relsType, dataFile, contentType, true);
                    var formatedFloatingLabels = floatingElement.Format(attributes);

                    using (var writer = new StreamWriter(floatingLabelsStream))
                    {
                        writer.Write(formatedFloatingLabels);
                        writer.Flush();
                    }
                }
            }
        }

        private static string ReadFloatingElement(Stream stream, string relsType, string dataFile, string contentType)
        {
            using (var package = Package.Open(stream, FileMode.Open, FileAccess.Read))
            {
                if (!XpsTools.IsDataFileExist(package, relsType, dataFile, contentType))
                    return string.Empty;

                using (var streamInPackage = XpsTools.GetStreamInPackage(package, relsType, dataFile, contentType, false))
                {
                    if (stream == null)
                        return string.Empty;
                    using (var reader = new StreamReader(streamInPackage))
                    {
                        var result = reader.ReadToEnd();
                        return result;
                    }
                }
            }
        }
    }
}
