/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Text.RegularExpressions;
using System.Windows;
using System.Windows.Media;
using System.Xml.Linq;

namespace Pilot.Xps.Domain.Labels
{
    class TextLabel
    {
        public TextLabel(string xml)
        {
            var normalTextBlockXaml = new TextBlockXamlHandler(xml);
            Value = normalTextBlockXaml.Value;

            xml = normalTextBlockXaml.GetNormalTextBlockXaml();

            var tmp = GetXElement(xml);
            TextLabelXElement = tmp;

            SetBaseSides();
        }

        private void SetBaseSides()
        {
            switch (HorizontalAlignment)
            {
                case HorizontalAlignment.Left:
                    if (string.IsNullOrEmpty(LeftValue))
                        Left = 0.0;
                    break;
                case HorizontalAlignment.Center:
                    break;
                case HorizontalAlignment.Right:
                    if (string.IsNullOrEmpty(RightValue))
                        Right = 0.0;
                    break;
                case HorizontalAlignment.Stretch:
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }

            switch (VerticalAlignment)
            {
                case VerticalAlignment.Top:
                    if (string.IsNullOrEmpty(TopValue))
                        Top = 0.0;
                    break;
                case VerticalAlignment.Center:
                    break;
                case VerticalAlignment.Bottom:
                    if (string.IsNullOrEmpty(BottomValue))
                        Bottom = 0.0;
                    break;
                case VerticalAlignment.Stretch:
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public XElement TextLabelXElement { get; }

        public TextWrapping TextWrapping
        {
            get => GetWrapAttribute(BarcodeLabelsConstants.TEXT_WRAP);
            set => SetWrapValue(BarcodeLabelsConstants.TEXT_WRAP, value);
        }

        public double FontSize
        {
            get => GetDoubleAttribute(BarcodeLabelsConstants.FONT_SIZE);
            set => SetAttributeValue(BarcodeLabelsConstants.FONT_SIZE, value);
        }

        public FontFamily FontFamily
        {
            get
            {
                var fontName = GetStringAttribute(BarcodeLabelsConstants.FONT_FAMILY);
                if (string.IsNullOrEmpty(fontName))
                {
                    return new FontFamily("Arial");
                }

                return new FontFamily(fontName);
            }
            set => SetFontAttributeValue(BarcodeLabelsConstants.FONT_FAMILY, value);
        }

        public HorizontalAlignment HorizontalAlignment
        {
            get => GetHorizontalAlignmentAttribute(BarcodeLabelsConstants.HORIZONTAL_ALIGNMENT);
            set => SetHorizontalAlignmentAttribute(BarcodeLabelsConstants.HORIZONTAL_ALIGNMENT, value);
        }

        public VerticalAlignment VerticalAlignment
        {
            get => GetVerticalAlignmentAttribute(BarcodeLabelsConstants.VERTICAL_ALIGNMENT);
            set => SetVerticalAlignmentAttribute(BarcodeLabelsConstants.VERTICAL_ALIGNMENT, value);
        }

        public bool Floating
        {
            get => GetBoolAttribute(BarcodeLabelsConstants.FLOATING_ATTRIBUTE);
            set => SetAttributeValue(BarcodeLabelsConstants.FLOATING_ATTRIBUTE, value);
        }

        public double Width
        {
            get => GetDoubleAttribute(BarcodeLabelsConstants.WIDTH);
            set => SetAttributeValue(BarcodeLabelsConstants.WIDTH, value);
        }

        public double Height
        {
            get => GetDoubleAttribute(BarcodeLabelsConstants.HEIGHT);
            set => SetAttributeValue(BarcodeLabelsConstants.HEIGHT, value);
        }

        public bool HideIfEmptyValue
        {
            get => GetBoolAttribute(BarcodeLabelsConstants.HIDE_IF_EMPTY_VALUE_ATTRIBUTE);
            set => SetAttributeValue(BarcodeLabelsConstants.HIDE_IF_EMPTY_VALUE_ATTRIBUTE, value);
        }
        public string PageRange
        {
            get => GetStringAttribute(BarcodeLabelsConstants.TEXT_LABEL_PAGE_RANGE);
            set => SetAttributeValue(BarcodeLabelsConstants.TEXT_LABEL_PAGE_RANGE, value);
        }
        public double Angle
        {
            get
            {
                var angle = GetDoubleAttribute(BarcodeLabelsConstants.ANGLE_ATTRIBUTE);
                if (double.IsNaN(angle))
                {
                    return 0.0;
                }

                return angle;
            }
            set => SetAttributeValue(BarcodeLabelsConstants.ANGLE_ATTRIBUTE, value);
        }

        public string TopValue
        {
            get => GetStringAttribute(BarcodeLabelsConstants.CANVAS_TOP);
            //set => SetOffsetAttribute(Constants.CANVAS_TOP, value);
        }

        public string LeftValue
        {
            get => GetStringAttribute(BarcodeLabelsConstants.CANVAS_LEFT);
            //set => SetOffsetAttribute(Constants.CANVAS_LEFT, value);
        }

        public string RightValue
        {
            get => GetStringAttribute(BarcodeLabelsConstants.CANVAS_RIGHT);
            //set => SetOffsetAttribute(Constants.CANVAS_RIGHT, value);
        }

        public string BottomValue
        {
            get => GetStringAttribute(BarcodeLabelsConstants.CANVAS_BOTTOM);
            //set => SetOffsetAttribute(Constants.CANVAS_BOTTOM, value);
        }

        public double Top
        {
            set => SetOffsetAttributeDouble(BarcodeLabelsConstants.CANVAS_TOP, value);
        }

        public double Left
        {
            set => SetOffsetAttributeDouble(BarcodeLabelsConstants.CANVAS_LEFT, value);
        }

        public double Right
        {
            set => SetOffsetAttributeDouble(BarcodeLabelsConstants.CANVAS_RIGHT, value);
        }

        public double Bottom
        {
            set => SetOffsetAttributeDouble(BarcodeLabelsConstants.CANVAS_BOTTOM, value);
        }

        public string Text
        {
            get => GetStringAttribute(BarcodeLabelsConstants.TEXT);
            set => SetAttributeValue(BarcodeLabelsConstants.TEXT, value);
        }

        public bool KeepWithinPageBounds
        {
            get => GetBoolAttribute(BarcodeLabelsConstants.KEEP_WITHIN_PAGE_BOUNDS_ATTRIBUTE);
        }

        public string Value { get; }

        public TextAlignment TextAlignment
        {
            get => GetTextAlignmentAttribute(BarcodeLabelsConstants.TEXT_ALIGNMENT);
            set => SetTextAlignmentAttribute(BarcodeLabelsConstants.TEXT_ALIGNMENT, value);
        }

        public void ClearOffsetsToLabel()
        {
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_LEFT);
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_TOP);
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_BOTTOM);
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_RIGHT);
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_CENTER_X);
            RemoveAttribute(TextLabelXElement, BarcodeLabelsConstants.CANVAS_CENTER_Y);
        }
        public static string GetWrappedSquareBrackets(string str)
        {
            var pattern = "(?<match>(\\{(.*?)\\}))";

            var replacePattern = "[${match}]";
            return Regex.Replace(str, pattern, replacePattern);
        }

        public static string GetUnwrappedSquareBrackets(string str)
        {
            var pattern = "(\\[\\{(?<match>(.*?))\\}\\])";

            var replacePattern = "{${match}}";
            return Regex.Replace(str, pattern, replacePattern);
        }

        public static void RemoveAttribute(XElement element, string attrName)
        {
            XAttribute att = element.Attribute(attrName);
            att?.Remove();
        }

        private XElement GetXElement(string xml)
        {
            var xElement = XElement.Parse(xml);
            return xElement;
        }

        private HorizontalAlignment GetHorizontalAlignmentAttribute(string attributeName)
        {
            var str = GetStringAttribute(attributeName);

            switch (str)
            {
                case "Left":
                    return HorizontalAlignment.Left;
                case "Right":
                    return HorizontalAlignment.Right;
                case "Center":
                    return HorizontalAlignment.Center;
                default:
                    return HorizontalAlignment.Left;
            }
        }

        private VerticalAlignment GetVerticalAlignmentAttribute(string attributeName)
        {
            var str = GetStringAttribute(attributeName);

            switch (str)
            {
                case "Top":
                    return VerticalAlignment.Top;
                case "Bottom":
                    return VerticalAlignment.Bottom;
                case "Center":
                    return VerticalAlignment.Center;
                default:
                    return VerticalAlignment.Top;
            }
        }

        private TextAlignment GetTextAlignmentAttribute(string attributeName)
        {
            var str = GetStringAttribute(attributeName);

            switch (str)
            {
                case "Left":
                    return TextAlignment.Left;
                case "Right":
                    return TextAlignment.Right;
                case "Center":
                    return TextAlignment.Center;
                case "Justify":
                    return TextAlignment.Justify;
                default:
                    throw new ArgumentOutOfRangeException(nameof(str), str, null);
            }
        }

        private void SetTextAlignmentAttribute(string attributeName, TextAlignment textAlignment)
        {
            switch (textAlignment)
            {
                case TextAlignment.Left:
                    SetAttributeValue(attributeName, "Left");
                    break;
                case TextAlignment.Right:
                    SetAttributeValue(attributeName, "Right");
                    break;
                case TextAlignment.Center:
                    SetAttributeValue(attributeName, "Center");
                    break;
                case TextAlignment.Justify:
                    SetAttributeValue(attributeName, "Justify");
                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(textAlignment), textAlignment, null);
            }
        }

        private void SetHorizontalAlignmentAttribute(string attributeName, HorizontalAlignment horizontalAlignment)
        {
            switch (horizontalAlignment)
            {
                case HorizontalAlignment.Left:
                    SetAttributeValue(attributeName, "Left");
                    break;
                case HorizontalAlignment.Center:
                    SetAttributeValue(attributeName, "Center");
                    break;
                case HorizontalAlignment.Right:
                    SetAttributeValue(attributeName, "Right");
                    break;
                case HorizontalAlignment.Stretch:
                    SetAttributeValue(attributeName, "Left");
                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(horizontalAlignment), horizontalAlignment, null);
            }
        }

        private void SetVerticalAlignmentAttribute(string attributeName, VerticalAlignment verticalAlignment)
        {
            switch (verticalAlignment)
            {
                case VerticalAlignment.Top:
                    SetAttributeValue(attributeName, "Top");
                    break;
                case VerticalAlignment.Center:
                    SetAttributeValue(attributeName, "Center");
                    break;
                case VerticalAlignment.Bottom:
                    SetAttributeValue(attributeName, "Bottom");
                    break;
                case VerticalAlignment.Stretch:
                    SetAttributeValue(attributeName, "Top");
                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(verticalAlignment), verticalAlignment, null);
            }
        }

        private TextWrapping GetWrapAttribute(string attributeName)
        {
            var str = GetStringAttribute(attributeName);

            switch (str)
            {
                case "WrapWithOverflow":
                    return TextWrapping.WrapWithOverflow;
                case "NoWrap":
                    return TextWrapping.NoWrap;
                case "Wrap":
                    return TextWrapping.Wrap;
                default:
                    return TextWrapping.NoWrap;
            }
        }

        private void SetWrapValue(string attributeName, TextWrapping textWrapping)
        {
            switch (textWrapping)
            {
                case TextWrapping.WrapWithOverflow:
                    SetAttributeValue(attributeName, "WrapWithOverflow");
                    break;
                case TextWrapping.NoWrap:
                    SetAttributeValue(attributeName, "NoWrap");
                    break;
                case TextWrapping.Wrap:
                    SetAttributeValue(attributeName, "Wrap");
                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(textWrapping), textWrapping, null);
            }
        }

        private bool GetBoolAttribute(string attributeName)
        {
            var attr = TextLabelXElement.Attribute(attributeName);
            if (attr != null)
            {
                return bool.Parse(attr.Value);
            }

            return false;
        }

        private string GetStringAttribute(string attributeName)
        {
            var attr = TextLabelXElement.Attribute(attributeName);
            return attr?.Value;
        }

        private double GetDoubleAttribute(string attributeName)
        {
            var attr = TextLabelXElement.Attribute(attributeName);
            if (attr == null)
                return default(double);

            return double.TryParse(attr.Value, out var number) ? number : default(double);
        }

        private void SetOffsetAttributeDouble(string attributeName, double value)
        {
            if (double.IsNaN(value))
                TextLabelXElement.Attribute(attributeName)?.Remove();
            else
                TextLabelXElement.SetAttributeValue(attributeName, value);
        }

        private void SetAttributeValue<T>(string attrName, T value)
        {
            var attr = TextLabelXElement.Attribute(attrName);
            if (attr != null)
            {
                TextLabelXElement.SetAttributeValue(attrName, value);
                return;
            }

            attr = new XAttribute(attrName, value);
            TextLabelXElement.Add(attr);
        }

        private void SetFontAttributeValue(string attrName, FontFamily fontFamily)
        {
            SetAttributeValue(attrName, fontFamily.ToString());
        }
    }
}
