/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;

namespace Pilot.Xps.Domain.Signatures
{
    class SignatureBuilder
    {
        private readonly IModifierBackend _backend;
        private readonly IXpsPartLoader _xpsPartLoader;
        private readonly IXpsSignatureSettings _settings;
        private readonly IPilotSignatureProxy _proxy;

        public SignatureBuilder(IModifierBackend backend, IXpsPartLoader xpsPartLoader)
        {
            _backend = backend;
            _xpsPartLoader = xpsPartLoader;
            _proxy = new PilotSignatureProxy(_backend);
            _settings = new XpsSignatureSettings(_proxy);
        }

        internal IList<DigitalSignature> BuildSignatureWithRequestForAllPositions(DocumentContext context, INObject document, IEnumerable<int> positionIds, X509Certificate2 certificate)
        {
            var stream = context.CreateStream(DocumentUsageArea.Signatures);
            context.GetStream(DocumentUsageArea.Original).CopyTo(stream);
            var signatures = _xpsPartLoader.DownloadSignatures(out var definitions);
            XpsSignatureHelper.AddDefinitions(definitions, stream);
            
            var errorCode = XpsSignatureHelper.MergeDigitalSignatures(signatures, stream);
            stream.Register();
            if (!errorCode.IsResultOk())
                throw new SignatureException("This document contains invalid digital signatures. Error: " + errorCode);

            var nSignatures = document.ActualFileSnapshot.Files.SelectMany(f => f.Signatures);
            var signaturesToSign = nSignatures.Where(v => positionIds.Contains(v.PositionId)).ToList();
            var files = document.ActualFileSnapshot.Files.ToList();
            var signatureDefinitions = GetSignatureDefinitionToSign(files, signaturesToSign, context);
            var virtualDefinitions = GetVirtualSignatureDisplayParams(files);
            var allSignatureDefinitions = virtualDefinitions.Union(signatureDefinitions).Where(p => _proxy.CanSign(document, p.SignId, positionIds)).ToList();
            return SignAndSaveToDatabase(allSignatureDefinitions, document, context, certificate);
        }

        internal IList<SignatureDisplayParams> BuildSignatures(DocumentContext context, IList<INFile> files)
        {
            var stream = context.CreateStream(DocumentUsageArea.Signatures);
            context.GetStream(DocumentUsageArea.Original).CopyTo(stream);
            var signatures = _xpsPartLoader.DownloadSignatures(out var definitions);
            XpsSignatureHelper.AddDefinitions(definitions, stream);

            var errorCode = XpsSignatureHelper.MergeDigitalSignatures(signatures, stream);
            stream.Register();
            if (!errorCode.IsResultOk())
                throw new SignatureException("This document contains invalid digital signatures. Error: " + errorCode);

            var nSignatures = files.SelectMany(f => f.Signatures);
            var signatureDefinitions = GetSignatureDefinitionToSign(files, nSignatures, context, false);
            var virtualDefinitions = GetVirtualSignatureDisplayParams(files);
            var allSignatureDefinitions = virtualDefinitions.Union(signatureDefinitions).ToList();
            return allSignatureDefinitions;
        }

        internal IList<DigitalSignature> SignAndSaveToDatabase(List<SignatureDisplayParams> signatureDefinitions, INObject document, DocumentContext context, X509Certificate2 certificate)
        {
            var mgr = new XpsSignatureManager(_settings, context);

            signatureDefinitions.ForEach(sd =>
            {
                sd.IsChecked = true;
                mgr.Signatures.Add(sd);
            });
            
            return mgr.SignInternal(document, certificate, false).ToList();
        }

        private IList<SignatureDisplayParams> GetSignatureDefinitionToSign(IList<INFile> files, IEnumerable<INSignature> signaturesToSign, DocumentContext context, bool withFilter = true)
        {
            var signerIds = signaturesToSign.Select(x => x.Id).ToList();
            var stream = context.GetStream(DocumentUsageArea.Signatures);
            var signatureDefinitions = _settings.GetSignatureDisplayParams(files, stream).ToList();
            if (withFilter)
                signatureDefinitions = signatureDefinitions.Where(x => signerIds.Contains(x.SignId ?? Guid.Empty)).ToList();

            return signatureDefinitions;
        }

        private void AddDefinitions(INObject currentFile, DocumentContext context)
        {
            var xpsSignaturesDefinitions = currentFile.ActualFileSnapshot.Files.First(x => FileExtensionHelper.IsXpsAlike(x.Name))
                .Signatures.Select(s => new XpsSignatureDefinition
                {
                    SpotId = s.Id,
                    RequestedSigner = s.RequestedSigner,
                    Intent = s.Role
                });
            var xpsStream = context.ModifyStream(DocumentUsageArea.Signatures);
            XpsSignatureHelper.AddDefinitions(xpsSignaturesDefinitions.ToList(), xpsStream);
            xpsStream.Register();
        }

        private IEnumerable<SignatureDisplayParams> GetVirtualSignatureDisplayParams(IList<INFile> files)
        {
            // все неподписанные дополнительные запросы на подпись
            return _xpsPartLoader.GetAdditionalDefinitions()
                .Where(d => string.IsNullOrEmpty(d.Sign))
                .Select(signature => new SignatureDisplayParams
                {
                    Certificate = null,
                    RequestedSigner = signature.RequestedSigner,
                    SignerId = XpsSignatureUtils.Id2Guid(_backend.GetDatabaseId(), signature.PositionId),
                    CanUserSign = _proxy.CanSign(files, signature.ToXpsSignatureDefinition()),
                    IsSigned = false,
                    IsValid = false,
                    IsChecking = true,
                    SignId = signature.Id,
                    Role = signature.Role,
                    IsAdditional = true
                });
        }
    }
}
