/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.DataClasses;

namespace Pilot.Xps.Domain.Signatures
{
    class XpsSignatureManager
    {
        public Collection<SignatureDisplayParams> Signatures { get; protected set; }

        public IXpsSignatureSettings Settings => _settings;
        private readonly IXpsSignatureSettings _settings;
        private readonly DocumentContext _documentContext;

        public XpsSignatureManager(IXpsSignatureSettings settings, DocumentContext context)
        {
            _settings = settings;
            _documentContext = context;
            Signatures = new Collection<SignatureDisplayParams>();
        }

        //public void Sign(X509Certificate2 certificate)
        //{
        //    if (certificate == null)
        //        _onSigningListener.OnSigningCompleted(true, null, null);
        //    var res = SignInternal(certificate, true);
        //    if (res == null)
        //        _onSigningListener.OnSigningCompleted(true, null, null);
        //    _onSigningListener.OnSigningCompleted(false, res, null);
        //}

        //public void SignWithoutRequest(BarcodeLabelContent barcodeLabelsContent, IList<GraphicLayerElement> graphicLayers, X509Certificate2 certificate)
        //{
        //    var stream = _documentContext.ModifyStream(DocumentUsageArea.Signatures);
        //    try
        //    {
        //        var resultStream = _onSigningListener.InjectContentBeforeSigning(_dispatcher, stream, graphicLayers, barcodeLabelsContent);
        //        var wasStreamUpdatedWhileInject = resultStream != stream;
        //        var digitalSignatures = SignNoRequests(resultStream, certificate, out bool wasStreamUpdatedWhileSigning);
        //        var embededLayers = graphicLayers.Where(l => !l.IsFloating).ToList();
        //        _onSigningListener.OnSigningWithoutRequestCompleted(false, digitalSignatures, embededLayers, barcodeLabelsContent,
        //            wasStreamUpdatedWhileInject || wasStreamUpdatedWhileSigning ? resultStream : null);
        //    }
        //    finally
        //    {
        //        stream.Register();
        //    }
        //}

        //public void RemoveUnsavedSignature(ObservableCollection<SignatureDisplayParams> signaturesCollection, DigitalSignature signature)
        //{
        //    var stream = _documentContext.ModifyStream(DocumentUsageArea.Signatures);
        //    using (var p = Package.Open(stream, FileMode.Open, FileAccess.Read))
        //    {
        //        using (var xpsDoc = new XpsDocument(p))
        //        {
        //            var packageSignatureManager = new PackageDigitalSignatureManager(p);
        //            var signaturesList = packageSignatureManager.Signatures.Select(packageDigitalSignature => new XpsDigitalSignature(packageDigitalSignature, xpsDoc)).ToList();
        //            foreach (var xpsDigitalSignature in signaturesList)
        //            {
        //                if (xpsDigitalSignature.Id == signature.SpotId)
        //                {
        //                    var certificate2 = new X509Certificate2(xpsDigitalSignature.SignerCertificate);
        //                    var item = signaturesCollection.Where(s => s.Certificate != null && s.Certificate.Equals(certificate2)
        //                        && s.SignDate.Equals(xpsDigitalSignature.SigningTime.ToLocalTime().ToString("g"))).ToList();
        //                    signaturesCollection.Remove(item.First());
        //                }
        //            }
        //        }
        //    }
        //}

        //List<DigitalSignature> SignNoRequests(Stream xpsStream, X509Certificate2 certificate, out bool streamWasUpdated)
        //{
        //    streamWasUpdated = false;
        //    try
        //    {
        //        XpsSignatureHelper.SignAsWithoutRequest(xpsStream, certificate, out streamWasUpdated);
        //        return _digitalExporter.ExportFromXps(xpsStream);
        //    }
        //    catch (CertificateNotFoundException)
        //    {
        //        throw;
        //    }
        //    catch (CryptographicException ex)
        //    {
        //        uint hresult = (uint)Marshal.GetHRForException(ex);
        //        if (hresult.Equals(0x8010006E))
        //            return _digitalExporter.ExportFromXps(xpsStream);
        //        throw;
        //    }
        //    catch (Exception ex)
        //    {
        //        uint hresult = (uint)Marshal.GetHRForException(ex);
        //        if (hresult.Equals(2148734229))
        //            throw new SignatureException(LocalizationResources.AlgorithmIsNotSupport, ex);
        //        throw new SignatureException(LocalizationResources.DocumentNotSignWasErrorMessage, ex);
        //    }
        //}

        public IList<DigitalSignature> SignInternal(INObject document, X509Certificate2 certificate, bool actualizeSignatures)
        {
            try
            {
                var filtered = FilterUncheked(Signatures);
                var signedParams = _settings.SignAs(document, filtered, _documentContext, certificate, actualizeSignatures);

                foreach (var signed in signedParams)
                    Signatures.Single(p => p.SignId.Equals(signed.SignId)).Assign(signed);

                return ExportFromXps(_documentContext.GetStream(DocumentUsageArea.Signatures));
            }
            catch (CertificateNotFoundException)
            {
                throw;
            }
            catch (CryptographicException ex)
            {
                uint hresult = (uint)Marshal.GetHRForException(ex);
                //действие отменено пользователем
                if (hresult.Equals(0x8010006E))
                    return ExportFromXps(_documentContext.GetStream(DocumentUsageArea.Signatures));
                throw;
            }
            catch (Exception ex)
            {
                uint hresult = (uint)Marshal.GetHRForException(ex);

                // Это ожидаемые ошибки
                if (hresult.Equals(2148734229))
                    throw new SignatureException("The algorithm is not supported by the certificate, may not set crypto provider", ex, true);

                if (hresult.Equals(2148735296))
                    throw new SignatureException("No root certificate", ex, true);

                // случилось что-то плохое, неважно по какой причине
                throw new SignatureException("Document was not signed due to internal error", ex);
            }
        }

        private IList<DigitalSignature> ExportFromXps(Stream xpsStream)
        {
            using (var xpsPackage = Package.Open(xpsStream, FileMode.Open, FileAccess.Read))
            {
                return XpsSignatureUtils.GetDigitalSignatures(xpsPackage);
            }
        }
        /// <summary>
        /// special code for default windows viewer
        /// </summary>
        //public static void MakeCompatibleWithDefaultWindowsViewer(Stream source, Stream dest)
        //{
        //    using (var p = Package.Open(source))
        //    using (var doc = new XpsDocument(p))
        //    {
        //        var defs = doc.GetSignatureDefinitions();
        //        XpsSignatureHelper.RemoveSignaturesAndDefinitions(dest);
        //        XpsSignatureHelper.AddDefinitions(defs.ToList(), dest);
        //    }
        //}

        //public void Dispose()
        //{
        //}

        private IList<SignatureDisplayParams> FilterUncheked(Collection<SignatureDisplayParams> signatures)
        {
            return signatures.Where(s => s.IsChecked && !s.IsSigned).ToList();
        }

        //private Collection<SignatureDisplayParams> Read(Stream xpsStream)
        //{
        //    Signatures = Settings.GetSignatureDisplayParams(xpsStream);
        //    return Signatures;
        //}
    }
}
