/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.Common;
using Pilot.Xps.Domain.Render;

namespace Pilot.Xps.Domain.Tools
{
    static class ImageHelper
    {
        /// <summary>
        /// Конвертирует изображение в png, если исходное изображение не поддерживается контейнером xps
        /// Многостраничные tiff конвертируются в набор одностраничных tiff (поэтому возвращается коллекция имён файлов)
        /// Если конвертация не требуется, то файл не изменяется и возвращается путь исходного файла
        /// </summary>
        /// <param name="imageFilename"></param>
        /// <param name="tempDirName"></param>
        /// <returns></returns>
        public static IEnumerable<string> ConvertImageToFormatGoodForXps(string imageFilename, string tempDirName)
        {
            if (!File.Exists(imageFilename))
            {
                throw new FileNotFoundException(String.Format("Файл с именем {0} не найден. Проверьте правильность аргументов командной строки.", imageFilename));
            }

            Image image;

            try
            {
                image = Image.FromFile(imageFilename);
            }
            catch (Exception ex)
            {
                throw new FileLoadException(String.Format("Не удалось загрузить изображение из файла \"{0}\". Возможно, файл повреждён, либо не является изображением.", imageFilename), ex);
            }

            using (image)
            {
                var dimensionId = image.FrameDimensionsList[0];
                var dimension = new FrameDimension(dimensionId);
                int noOfPages = image.GetFrameCount(dimension);

                var encodeInfo = GetImageCodecInfo(image);

                if (encodeInfo == null)
                    throw new FileFormatException(
                        String.Format("Формат изображения не поддерживается. FormatId = {0}, файл \"{1}\"",
                                      image.RawFormat.Guid, imageFilename));

                if (ImageFormat.Bmp.Equals(image.RawFormat))
                {
                    // Преобразуем bmp в png

                    var pngEncoder = ImageCodecInfo.GetImageEncoders().FirstOrDefault(enc => enc.FormatID.Equals(ImageFormat.Png.Guid));

                    if (pngEncoder == null)
                        throw new FileFormatException("В системе не найден кодек формата png"); // но не будет такого

                    var savePath = GetTempFileName(tempDirName);

                    image.Save(savePath, pngEncoder, null);
                    return new[] { savePath };
                }

                if (image.FrameDimensionsList.Length <= 1 && noOfPages <= 1)
                    return new[] { imageFilename }; // в изображении один кадр - возвращаем имя исходного файла


                // Разбиваем на страницы
                var result = new List<string>();

                EncoderParameters eps = null;

                // Если это действительно tiff (вряд-ли что-то ещё), и 1 бит на писель, задём компрессию CCITT
                if (encodeInfo.FormatID.Equals(ImageFormat.Tiff.Guid) && (image.PixelFormat == PixelFormat.Format1bppIndexed))
                {
                    eps = new EncoderParameters(1);
                    eps.Param[0] = new EncoderParameter(Encoder.Compression, (long)EncoderValue.CompressionCCITT4);
                }

                foreach (var guid in image.FrameDimensionsList)
                {
                    for (var index = 0; index < noOfPages; index++)
                    {
                        var savePath = GetTempFileName(tempDirName);

                        var currentFrame = new FrameDimension(guid);
                        image.SelectActiveFrame(currentFrame, index);
                        image.Save(savePath, encodeInfo, eps);

                        result.Add(savePath);
                    }
                }

                return result;
            }
        }

        public static ImageInfo GetImageInfo(string imageFilename)
        {
            using (var image = Image.FromFile(imageFilename))
            {
                var encodeInfo = GetImageCodecInfo(image);

                if (encodeInfo == null)
                    throw new FileFormatException(
                        String.Format("Формат изображения не поддерживается. FormatId = {0}, файл \"{1}\"",
                                      image.RawFormat.Guid, imageFilename));

                XpsImageType xpsImageType;

                if (ImageFormat.Jpeg.Equals(image.RawFormat))
                {
                    xpsImageType = XpsImageType.JpegImageType;
                }
                else

                    if (ImageFormat.Png.Equals(image.RawFormat))
                {
                    xpsImageType = XpsImageType.PngImageType;
                }
                else

                        if (ImageFormat.Tiff.Equals(image.RawFormat))
                {
                    xpsImageType = XpsImageType.TiffImageType;
                }
                else
                    throw new FileFormatException(
                        String.Format("Формат изображения не поддерживается в xps-документах. FormatId = {0}, файл \"{1}\"",
                                      image.RawFormat.Guid, imageFilename));

                return new ImageInfo
                {
                    XpsImageType = xpsImageType,
                    Width = image.Size.Width,
                    Height = image.Size.Height,
                    HorizontalResolution = image.HorizontalResolution,
                    VerticalResolution = image.VerticalResolution,
                    Uri = imageFilename
                };
            }
        }

        public static ImageCodecInfo GetImageCodecInfo(Image image)
        {
            var imageEncoders = ImageCodecInfo.GetImageEncoders();

            return imageEncoders.FirstOrDefault(t => t.FormatID == image.RawFormat.Guid);
        }

        private static string GetTempFileName(string tempDirName)
        {
            string savePath;
            do
            {
                savePath = Path.Combine(tempDirName, Path.GetRandomFileName());
            } while (File.Exists(savePath));

            return savePath;
        }

        /// <summary>
        /// Создаёт временную директорию в папке %Temp% со случайным сгенерированным названием
        /// </summary>
        /// <param name="prefix">Префикс, который будет добавлен в начало имени папки</param>
        /// <returns></returns>
        public static string CreateTempDir(string prefix = "")
        {
            string tempDir;
            string tempPath = GetTempPath("Pilot-XpsRender");
            do
            {
                tempDir = Path.Combine(tempPath, prefix + Path.GetRandomFileName().Replace(".", string.Empty));
            }
            while (Directory.Exists(tempDir));
            Directory.CreateDirectory(tempDir);

            return tempDir;
        }

        private static string GetTempPath(string productName)
        {
            var tempPath = Path.Combine(SpecialDirectoryProvider.GetTempPath(), productName);
            if (!Directory.Exists(tempPath))
                Directory.CreateDirectory(tempPath);
            return tempPath;
        }

    }
}
