/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Ascon.Pilot.Server.Api.Contracts;
using Ascon.Pilot.Transport;

namespace Pilot.Xps.Connection.Client
{
    public interface IRepository
    {
        T GetServerCommandProxy<T>(string commandProcessorName) where T : class;
        Task<ServerCommandRequestResult> InvokeServerCommand(string commandName, byte[] data);
    }

    public class Repository : IRepository, ICallbackListener
    {
        private readonly IServerApi _serverApi;
        private readonly Dictionary<Guid, TaskCompletionSource<ServerCommandRequestResult>> _serverCommands = new Dictionary<Guid, TaskCompletionSource<ServerCommandRequestResult>>();

        public Repository(IServerApi serverApi)
        {
            _serverApi = serverApi;
        }

        public T GetServerCommandProxy<T>(string commandProcessorName) where T : class
        {
            var callService = new ServerCommandCallService<T>(this, commandProcessorName);
            return new Marshaller(callService).Get<T>();
        }

        public Task<ServerCommandRequestResult> InvokeServerCommand(string commandName, byte[] data)
        {
            var requestId = Guid.NewGuid();
            var tcs = new TaskCompletionSource<ServerCommandRequestResult>();
            _serverCommands[requestId] = tcs;
            InvokeServerCommand(commandName, requestId, data);
            return tcs.Task;
        }

        private void InvokeServerCommand(string commandName, Guid requestId, byte[] data)
        {
            try
            {
                _serverApi.InvokeCommand(commandName, requestId, data);
            }
            catch (Exception e)
            {
                UpdateCommandError(requestId, e);
            }
        }

        private void UpdateCommandResult(Guid requestId, byte[] data, ServerCommandResult result)
        {
            if (_serverCommands.TryGetValue(requestId, out var tcs))
                tcs.SetResult(new ServerCommandRequestResult(data, (ServerCommandResult)(int)result));
        }

        private void UpdateCommandError(Guid requestId, Exception exception)
        {
            if (_serverCommands.TryGetValue(requestId, out var tcs))
                tcs.SetException(exception);
        }

        void ICallbackListener.UpdateCommandResult(Guid requestId, byte[] data, ServerCommandResult result)
        {
            UpdateCommandResult(requestId, data, result);
        }
    }
}
