/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Threading;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api;
using Ascon.Pilot.Transport;

namespace Pilot.Xps.Connection.Server
{
    public interface IConnectionService
    {
        bool IsConnected { get; }
        void Connect();
        void FirstConnect();
        void Disconnect();
        void ForceOffline();
    }

    public interface IConnectionListener
    {
        void Connected();
        void Disconnected(Exception exception = null);
    }

    internal class ConnectionService : IConnectionService, IConnectionLostListener
    {
        private const int RECONNECT_TIMEOUT = 5000;
        private readonly HttpPilotClient _client;
        private readonly ConnectionCredentials _credentials;
        private readonly IConnectionListener _connectionListener;
        private readonly IConnectionSettings _connectionSettings;
        private readonly object _locker = new object();
        private bool _isForceDisconnected = false;

        public ConnectionService(HttpPilotClient client, ConnectionCredentials credentials, IConnectionListener connectionListener, IConnectionSettings connectionSettings)
        {
            _client = client;
            _credentials = credentials;
            _connectionListener = connectionListener;
            _connectionSettings = connectionSettings;
            client.SetConnectionLostListener(this);
            IsConnected = false;
        }

        public bool IsConnected { get; private set; }

        public void Connect()
        {
            _isForceDisconnected = false;
            while (!IsConnected)
            {
                try
                {
                    if (_isForceDisconnected)
                    {
                        Disconnect();
                        return;
                    }

                    ConnectionProcess();
                }
                catch (Exception ex)
                {
                    if (ReconnectCouldHelp(ex))
                    {
                        Thread.Sleep(RECONNECT_TIMEOUT);
                        continue;
                    }
                    throw;
                }
            }
        }

        public void FirstConnect()
        {
            ConnectionProcess();
        }

        public void Disconnect()
        {
            lock (_locker)
            {
                try
                {
                    _client.Disconnect();
                    IsConnected = false;
                    _connectionListener.Disconnected();
                }
                catch (Exception ex)
                {
                    _connectionListener.Disconnected(ex);
                }
            }
        }

        public void ForceOffline()
        {
            _isForceDisconnected = true;
            IsConnected = false;
        }

        public void ConnectionLost(Exception ex = null)
        {
            lock (_locker)
            {
                IsConnected = false;
                //_connectionListener.Disconnected(ex);
                Connect();
            }
        }

        public static bool ReconnectCouldHelp(Exception ex)
        {
            return ex is TransportException || ex is PilotServerOfflineException || ex is UnauthorizedAccessException || ex is TimeoutException;
        }

        private void ConnectionProcess()
        {
            if (!IsConnected)
            {
                lock (_locker)
                {
                    if (!IsConnected)
                    {
                        _client.Connect(false);
                        var authenticationApi = _client.GetAuthenticationApi();
                        authenticationApi.Login(_credentials.DatabaseName, _credentials.Username, _credentials.ProtectedPassword, _credentials.UseWindowsAuth, _connectionSettings.LicenseCode);
                        IsConnected = true;
                        _connectionListener.Connected();
                    }
                }
            }
        }
    }
}
